// @ts-nocheck
const vscode = require('vscode');
const fs = require('fs');
const path = require('path');
const utils = require('./util');

const avmComponent = require('./lowcode/avm-component.json');
const basicComponent = require('./lowcode/basic-component.json');
const featureComponent = require('./lowcode/feature-components.json');
const layoutComponent = require('./lowcode/layout-components.json');

class StmlVisualize {
	constructor(context) {
		this.context = context;
	}

	static register(context) {
		// context.globalState.update()
		const provider = new StmlVisualize(context);
		const providerRegistration = vscode.window.registerCustomEditorProvider('apivisulize-stml', provider, {
			webviewOptions: {
				retainContextWhenHidden: true,
				supportsMultipleEditorsPerDocument: true,
				localResourceRoots: [vscode.Uri.file(path.join(context.extensionPath, 'webView'))]
			}
		});
		return providerRegistration;
	};
	static viewType = 'apivisulize-stml';

	async resolveCustomTextEditor(document, webviewPanel) {
		webviewPanel.webview.options = {
			enableScripts: true
		};
		this.getHtmlForWebview(webviewPanel, document.uri);

		function updateWebview() {
			webviewPanel.webview.postMessage({
				type: 'update'
			});
		}

		const changeDocumentSubscription = vscode.workspace.onDidChangeTextDocument(e => {
			if (e.document.uri.toString() === document.uri.toString()) {
				updateWebview();
			}
		});

		webviewPanel.onDidDispose(() => {
			changeDocumentSubscription.dispose();
		});

		webviewPanel.webview.onDidReceiveMessage(async cmdObj => {
			switch (cmdObj.cmd) {
				case 1: //可视化按钮切换成代码模式；
					this.openTextDocument(document.uri);
					break;
				case 2: //生成代码
					this.generateCode(cmdObj, webviewPanel);
					break;
				case 3:
					this.initPageContent(webviewPanel, document.uri);
					break;
				case 4://修改json文件
					this.modifyJson(cmdObj);
					break;
				case 5:
					vscode.window.showOpenDialog({
						canSelectFiles: true,
						canSelectFolders: false,
						canSelectMany: false,
						filters: [
							{ name: 'Image', extensions: ['jpg', 'jpe', 'jpeg', 'png', 'bmp', 'gif', 'ico', 'webp', 'avif'] }
						]
					}).then(async uris => {
						if (uris) {
							let uri = uris[0].fsPath;
							let stmlPath = cmdObj['data'].stmlPath;
							let compId = cmdObj['data'].id;

							let url = path.relative(path.dirname(stmlPath), uri);
							let message = {
								cmd: 6,
								data: {
									selectImage: url.replaceAll('\\', '/'),
									compId: compId,
									stmlPath: stmlPath
								}
							};
							webviewPanel.webview.postMessage(message);
						}
					});
					break;
				case 6:
					let message1 = {
						cmd: cmdObj['cmd'],
						data: {
							selectImage: cmdObj.uri.replaceAll('\\', '/'),
							compId: cmdObj['compId'],
							stmlPath: cmdObj['stmlPath']
						}
					};
					webviewPanel.webview.postMessage(message1);
					break;
				case 7://心跳包？？？
					webviewPanel.webview.postMessage(cmdObj);
					break;
				case 8: //导入组件
					// this.importComponent(cmdObj);
					console.log('cmd:8');
					break;
				// case 9: //埋点
				// 	vscode.commands.executeCommand('apicloud.lowcode.setEventKey',cmdObj);
				// 	break;
				case 10: {//加载本地图片
					let stmlPath = cmdObj.data.stmlPath;
					let imageUri = cmdObj.data.imageSrc;
					let absUrl = path.resolve(path.dirname(stmlPath), imageUri);
					if (fs.existsSync(absUrl) && fs.statSync(absUrl).isFile()) {
						webviewPanel.webview.postMessage(cmdObj)
					}
				}
					break;
				case 15: {//image 选取本地文件
					let filters;
					let canSelectFiles = true;
					switch (cmdObj['data']['type']) {
						case 'all':
							filters = []; break;
						case 'image':
							filters = [{ name: 'Image', extensions: ['png', 'jpg', 'jpeg'] }]; break;
						case 'file':
							filters = [{ name: 'File', extensions: ['doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'pdf', 'txt'] }]; break;
						case 'audio':
							filters = [{ name: 'Audio', extensions: ['mp3', 'wav', 'm4a'] }]; break;
						case 'video':
							filters = [{ name: 'Video', extensions: ['mp4', 'avi'] }]; break;
						default: break;
					}
					vscode.window.showOpenDialog({
						canSelectFiles,
						canSelectFolders: false,
						canSelectMany: false,
						filters: filters
					}).then(async uris => {
						if (uris) {
							const uri = uris[0].fsPath;
							const stmlPath = cmdObj['data'].stmlPath;
							const url = path.relative(path.dirname(stmlPath), uri);
							const cmdObj3 = {
								cmd: 15,
								uri: url
							}
							webviewPanel.webview.postMessage(cmdObj3);
						}
					});
				}
					break;

				case 11:
					vscode.window.showOpenDialog({
						canSelectFiles: true,
						canSelectFolders: false,
						canSelectMany: false,
						filters: [
							{ name: 'Image', extensions: ['jpg', 'jpe', 'jpeg', 'png', 'bmp', 'gif', 'ico', 'webp', 'avif'] }
						]
					}).then(async uris => {
						if (uris) {
							const uri = uris[0].fsPath;
							const stmlPath = cmdObj['data'].stmlPath;
							const compId = cmdObj['data'].id;
							const attrName = cmdObj['data'].attrName;

							const url = path.relative(path.dirname(stmlPath), uri);
							const option = {
								cmd: cmdObj['cmd'],
								data: {
									selectImage: url.replaceAll('\\', '/'),
									compId: compId,
									attrName: attrName
								}
							};
							webviewPanel.webview.postMessage(option);
						}
					});
					break;
				case 12: {//撤销
					const urlfile = cmdObj['data']['stmlPath'];
					vscode.commands.executeCommand('undo').then(() => {//撤销命令ID
						// vscode.workspace.textDocuments[0].save();
						return vscode.commands.executeCommand('workbench.action.files.save');//保存文件命令ID
					}).then(() => {
						let undoJson = fs.readFileSync(urlfile, { encoding: 'utf-8' });
						const cmdObj2 = {
							cmd: 12,
							data: {
								content: undoJson,
							}
						}
						webviewPanel.webview.postMessage(cmdObj2);
					});
				}
					break;
				case 13: {//恢复
					const redoFile = cmdObj['data']['stmlPath'];
					vscode.commands.executeCommand('redo').then(() => {//恢复命令ID
						return vscode.commands.executeCommand('workbench.action.files.save');//保存文件命令ID
					}).then(() => {
						let redojson = fs.readFileSync(redoFile, { encoding: 'utf-8' });
						const undoredoOpt = {
							cmd: 12,
							data: {
								content: redojson,
							}
						};
						webviewPanel.webview.postMessage(undoredoOpt);
					});
				}
					break;
				case 14://自定义组件更新属性
				case 16: //自定义组件事件写入
					this.getWriteFileCust(cmdObj);
					this.initPageContent(webviewPanel, document.uri);
					break;
				case 17: {
					let compPath = path.join(cmdObj.data.project.uri.fsPath, 'components');
					let obj = {};
					this.walkDir(compPath, cmdObj.data.project.uri.fsPath, obj);
					let message2 = {
						cmd: cmdObj['cmd'],
						data: {
							jsonData: obj
						}
					};
					webviewPanel.webview.postMessage(message2);
				}
					break;
				case 18:
					//从vue收到的命令
					//获取是否完成新手引导标记
					if (cmdObj.data.eventKey === 'getNewGuide') {
						// 获取值有问题
						vscode.commands.executeCommand('apicloud.lowcode.start.newguid').then((value) => {
							let startNewGuide = value;
							const message4 = {
								cmd: cmdObj['cmd'],
								data: {
									newGuide: startNewGuide ? startNewGuide : false
								}
							};
							webviewPanel.webview.postMessage(message4);
						})
					}
					//设置完成新手引导标记
					if (cmdObj.data.eventKey === 'setNewGuide') {
						vscode.commands.executeCommand('apicloud.lowcode.setnewguid');
					}

					break;
				default:
					break;

			}



		});

		updateWebview();
	}

	getHtmlForWebview(webviewPanel, uri) {
		this.filePath = uri.fsPath;
		webviewPanel.webview.html = utils.getWebViewContent(this.context, 'lowcode/dist/index.html');
	}
	// 打开配置文件源码
	openTextDocument(uri) {
		// vscode.commands.executeCommand('apicloud.lowcode.cmd.opencode',uri);
		vscode.commands.executeCommand('workbench.apicloud.open.lowcode', uri.fsPath);

	}
	// 初始化文件数据
	initPageContent(webviewPanel, uri) {
		let cmdObj = {
			cmd: 3,
			uri,
		}

		const project = utils.getWorkspaceFolder(uri.fsPath);
		if (project) {
			const uris = this.getUris(path.join(project['uri']['fsPath'], 'pages'), cmdObj['uri']);
			const config = this.getStmlConfig(cmdObj['uri']);
			const stmlData = fs.readFileSync(cmdObj['uri']['fsPath'], 'utf-8');
			if (!stmlData) {//拖入自定义i组件白屏的问题，定位到这里读文件返回了空
				return;
			}
			let custCompType = this.getComputedType(project, cmdObj['uri']['fsPath']);
			let packagejson;
			if (custCompType === 'custcomponent') {
				const fileName = utils.basename(cmdObj['uri']['fsPath'], '.stml');
				const componentsPath = path.join(project['uri']['fsPath'], `components/${fileName}/package.json`);
				packagejson = JSON.parse(fs.readFileSync(componentsPath, { encoding: 'utf-8' }));
			}
			// 获取新手引导
			let completeNewGuide = '';
			vscode.commands.executeCommand('apicloud.lowcode.start.newguid').then((value) => {
				completeNewGuide = value;
			}).then(() => {
				const componentMap = this.getCompontentList(project, stmlData);
				const message = {
					cmd: cmdObj['cmd'],
					data: {
						stmlPath: cmdObj['uri']['fsPath'].replaceAll('\\', '/'),
						project,
						stmlData,
						pages: uris,
						config,
						custCompType,
						packagejson,
						componentMap,
						completeNewGuide
					}
				};
				webviewPanel.webview.postMessage(message);
			});
		} else {
			const log = {
				level: 'notice-err',
				logMsg: '项目加载失败，请重启工具'
			};
			vscode.window.showErrorMessage(log.logMsg);
		}
	}

	getStmlConfig(uri) {
		let config = "{}";
		const jsonPath = path.join(uri.fsPath).replace('stml', 'json');
		if (fs.existsSync(jsonPath)) {
			config = fs.readFileSync(jsonPath, { encoding: 'utf-8' });
		}
		return config;
	}

	getComputedType(project, fsPath) {
		const fileName = utils.basename(fsPath, '.stml');
		const url = fsPath.replaceAll('\\', '/');
		const componentsPath = path.join(project.uri.fsPath, `components/${fileName}`).replaceAll('\\', '/');
		if (url.indexOf(componentsPath) > -1) {
			try {
				fs.accessSync(path.join(componentsPath, 'package.json'), fs.constants.F_OK);
				return 'custcomponent';
			} catch {
				return 'component';
			}
		} else {
			return 'page';
		}
	}

	getCompontentList(project, stmlData) {
		// 1.正则匹配获取获取当前页面import语句
		let importList = [], result;

		let componentObject = {}
		const re = /import\s*['"](.*?stml)['"]/g;
		while (result = re.exec(stmlData)) {
			importList.push(result[1]);
		}
		// 2.判断当前引入的组件是否为自定义组件，如果是，读取组件源码并存储于map中，并返回map
		importList.forEach(item => {
			let componentName = item.split('/').slice(-1)[0].split('.')[0];
			if (!(avmComponent[componentName] || basicComponent[componentName] || featureComponent[componentName] || layoutComponent[componentName])) {
				// 拼接自定义组件全路径，读取文件
				const componentsPath = path.join(project.uri.fsPath, item.slice(item.indexOf('components')));
				let custCompType = this.getComputedType(project, componentsPath);
				if (custCompType === 'custcomponent') {
					componentObject[componentName] = {
						template: fs.readFileSync(componentsPath, { encoding: 'utf-8' }) || ''
					};
					try {
						fs.accessSync(componentsPath.replace(`${componentName}.stml`, 'package.json'), fs.constants.F_OK);
						let packageJson = JSON.parse(fs.readFileSync(componentsPath.replace(`${componentName}.stml`, 'package.json'), { encoding: 'utf-8' }))
						componentObject[componentName]['meta'] = packageJson?.avm?.meta;
					} catch (err) {
						console.log('err', err);
					}
				}
			}

		});
		return componentObject;
	}

	getUris(project, uri) {
		// 读取当前项目page下的所有stml文件并获取其相对路径；
		let fileList = [];
		let files = fs.readdirSync(project);
		files.forEach(item => {
			if (item.includes('.stml')) {
				// 获取当前文件路径，并组织成相对路径；
				let itemFilePath = path.join(project, item);
				if (uri.fsPath != itemFilePath) {
					let filePath = path.relative(path.dirname(uri.fsPath), itemFilePath);
					if (filePath.indexOf('\\') < 0 && filePath.indexOf('/') < 0) {
						filePath = './' + filePath;
					}
					fileList.push(filePath.replaceAll('\\', '/'));
				}
			} else {
				if (!item.includes('.')) {
					fileList = fileList.concat(this.getUris(path.join(project, item), uri));
				}
			}
		});
		return fileList;
	}
	walkDir(dir, projectPath, obj) {
		let list = fs.readdirSync(dir);
		if (list.length) {
			for (let item of list) {
				if (item !== 'avm-ui' && item !== 'feature-component') {
					const itemPath = path.join(dir, item).replaceAll('\\', '/');
					let stats = fs.statSync(itemPath);
					if (stats.isDirectory()) {
						this.walkDir(itemPath, projectPath, obj);
					} else {
						if (!fs.existsSync(path.join(projectPath, item))) {
							let folderName = itemPath.split('/')[itemPath.split('/').length - 2];
							if (item === 'package.json' && fs.existsSync(itemPath.replace('package.json', folderName + '.stml'))) {
								//两个文件同时存在才是自定义组件
								let packageJson = JSON.parse(fs.readFileSync(itemPath, { encoding: 'utf-8' }));
								obj[folderName] = packageJson.avm.meta;
								obj[folderName]['group'] = '自定义组件';
								obj[folderName]['desc'] = packageJson.description;
								obj[folderName]['title'] = packageJson.name;
							}
						}
					}
				}
			}
		}
	}
	generateCode(receiveCmd) {//生成代码
		const content = receiveCmd['data'].content;
		const stmlPath = receiveCmd['data'].stmlPath;
		if (receiveCmd['data'].extra) {
			const extraObj = receiveCmd['data'].extra;
			const componentData = extraObj['componentData'];
			const resources = extraObj['resources'];
			fs.writeFileSync(stmlPath, content, { encoding: 'utf-8' });

			const p2 = this.copyComponent(componentData['type'], componentData['project']['uri']['fsPath'], resources);
			Promise.all([p2]).then(() => {
				const opt = {
					syncMode: 'auto-live-preview',
					uri: stmlPath,
					fileList: [stmlPath]
				};
				if (stmlPath.indexOf('/pages/') > -1) {
					vscode.commands.executeCommand('apicloud.lowcode.live.preview', opt);
				}
			});

			// const dataEvent = extraObj['dataEvent'];
			// const opt = {
			// 	cmd: 9,
			// 	data: dataEvent
			// };
			// vscode.commands.executeCommand('apicloud.lowcode.webview.cmd', opt);
		} else {
			fs.writeFile(stmlPath, content, () => {
				const opt = {
					syncMode: 'auto-live-preview',
					uri: stmlPath,
					fileList: [stmlPath]
				};
				if (stmlPath.indexOf('/pages/') > -1) {
					vscode.commands.executeCommand('apicloud.lowcode.live.preview', opt);
				}
			});
		}
	}
	copyComponent(type, project, resources) {
		return new Promise((resolve) => {
			if (type.indexOf('avm-') === 0) {
				let folderName = type.substring('avm-'.length);
				if (folderName.indexOf('-group') >= 0) {
					folderName = folderName.substring(0, folderName.lastIndexOf('-group'));
				}

				const avmPromise = this.copyComponnetPromise(project, 'avm-ui', folderName);
				const iconPromise = this.copyComponnetPromise(project, 'avm-ui', 'icon');

				Promise.all([avmPromise, iconPromise]).then(() => {
					resolve();
				});
			} else if (type.indexOf('hoc-') === 0) {
				const targetFile = path.join(project, 'components/feature-component');
				if (!fs.existsSync(path.dirname(targetFile))) {
					fs.mkdirSync(path.dirname(targetFile), { recursive: true });
				}
				const folder = type.substring(type.lastIndexOf('-') + 1);

				const p1 = this.copyFolder(path.join(__dirname, 'lowcode/feature-component', folder), targetFile, true);
				const p2 = this.copyResources(resources, project, folder);
				Promise.all([p1, p2]).then(() => {
					resolve();
				});
			} else {
				resolve();
			}
		});
	}
	copyResources(resources, project, folder) {
		return new Promise((resolve) => {
			const pros = resources.filter((res) => {
				const source = path.join(__dirname, 'lowcode/feature-component', folder, res);
				const target = path.join(project, res);
				if (fs.existsSync(source)) {
					if (!fs.existsSync(path.dirname(target))) {
						fs.mkdirSync(path.dirname(target), { recursive: true });
					}
					return this.copyFolder(source, target, true)
				} else {
					return;
				}
			});
			Promise.all(pros).then(() => {
				resolve();
			});
		});
	}
	copyComponnetPromise(project, compFolder, folder) {
		return new Promise((resolve) => {
			if (fs.existsSync(path.join(__dirname, `lowcode`, compFolder, folder))) {
				this.canCopy(project, folder, 'avm-ui').then(can => {
					if (can) {
						const targetFolder = path.join(project, 'components/avm-ui', folder);
						// 复制组件到项目当中；
						if (!fs.existsSync(path.join(project, 'components/avm-ui'))) {
							fs.mkdirSync(path.join(project, 'components/avm-ui'));
						}
						this.copyFolder(path.join(__dirname, 'lowcode', compFolder, folder), targetFolder, true);
						resolve();
					} else {
						resolve();
					}
				});
			} else {
				resolve();
			}
		});
	}
	async canCopy(project, folderName, type) {
		const targetFolder = path.join(project, 'components', type, folderName);
		if (!fs.existsSync(targetFolder)) { return true; }
		if (!fs.existsSync(path.join(targetFolder, 'version'))) { return true; }
		const content = await fs.readFile(path.join(path.join(targetFolder, 'version')), { encoding: 'utf-8' });
		if (content !== vscode.version) { return true; }

		return false;
	}

	getWriteFileCust(receiveCmd) {
		const data = receiveCmd.data;
		const stmlPath = data['stmlPath'];
		const fileName = utils.basename(stmlPath, '.stml');
		const jsonPath = path.join(path.dirname(stmlPath), `package.json`);
		receiveCmd.uri = stmlPath;
		const configjson = JSON.parse(fs.readFileSync(jsonPath, 'utf-8') || "{}");
		switch (receiveCmd['cmd']) {
			case 14:
				if (configjson['avm']['meta'] && JSON.stringify(configjson['avm']['meta']) !== '{}') {
					configjson['avm']['meta']['props'] = data.attrsList;
				} else {
					configjson['avm']['meta'] = {
						component: this.underline2Hump(fileName),
						props: data.attrsList,
						type: '2'
					};
				}
				break;
			case 16:
				if (configjson['avm']['meta'] && JSON.stringify(configjson['avm']['meta']) !== '{}') {
					configjson['avm']['meta']['event'] = data.eventList;
				} else {
					configjson['avm']['meta'] = {
						component: this.underline2Hump(fileName),
						enent: data.eventList,
						type: '2'
					};
				}
				break;
			default:
				break;
		}
		fs.writeFileSync(jsonPath, JSON.stringify(configjson, null, '\t'), { encoding: 'utf-8' });
	}
	underline2Hump(str) {
		let arr = str.split('-'); //["daaa", "df"] 或 ["", "daaa", "df"]
		for (let i = 0; i < arr.length; i++) {
			arr[i] = arr[i].slice(0, 1).toUpperCase() + arr[i].slice(1)
		}
		return arr.join('');
	}
	modifyJson(receiveCmd) {
		const data = receiveCmd.data;
		const stmlPath = data['stmlPath'];
		const bgColor = data['bgColor'];
		const fileName = utils.basename(stmlPath, '.stml');
		const jsonPath = path.join(path.dirname(stmlPath), `${fileName}.json`);
		if (fs.existsSync(jsonPath)) {
			const content = JSON.parse(fs.readFileSync(jsonPath, { encoding: 'utf-8' }));
			content['bgColor'] = bgColor;
			fs.writeFileSync(jsonPath, JSON.stringify(content, null, '\t'), { encoding: 'utf-8' });
		} else {
			const content = {
				bgColor
			};
			fs.writeFileSync(jsonPath, JSON.stringify(content, null, '\t'), { encoding: 'utf-8' });
		}
	}


	copyFolder(copiedPath, resultPath, direct) {
		if (!direct) {
			copiedPath = path.join(__dirname, copiedPath)
			resultPath = path.join(__dirname, resultPath)
		}

		function createDir(dirPath) {
			if (!fs.existsSync(dirPath)) {
				fs.mkdirSync(dirPath)
			}
		}

		if (fs.existsSync(copiedPath)) {
			createDir(resultPath)
			/**
			 * @des 方式一：利用子进程操作命令行方式
			 */
			// child_process.spawn('cp', ['-r', copiedPath, resultPath])

			/**
			 * @des 方式二：
			 */
			const files = fs.readdirSync(copiedPath, { withFileTypes: true });
			for (let i = 0; i < files.length; i++) {
				const cf = files[i]
				const ccp = path.join(copiedPath, cf.name)
				const crp = path.join(resultPath, cf.name)
				if (cf.isFile()) {
					/**
					 * @des 创建文件,使用流的形式可以读写大文件
					 */
					const readStream = fs.createReadStream(ccp)
					const writeStream = fs.createWriteStream(crp)
					readStream.pipe(writeStream)
				} else {
					try {
						/**
						 * @des 判断读(R_OK | W_OK)写权限
						 */
						fs.accessSync(path.join(crp, '..'), fs.constants.W_OK)
						this.copyFolder(ccp, crp, true)
					} catch (error) {
						console.log('folder write error:', error);
					}

				}
			}
		} else {
			console.log('do not exist path: ', copiedPath);
		}
	}


}

module.exports = StmlVisualize;


