/* Copyright 2020 APICloud Inc. All Rights Reserved.
 

 */

'use strict';

const { 
	warn
} = require('./client');

const {
	pathExists, 
	copyDirSync,
	copyFileSync,
	mkpath,
	scanStmlSync
} = require('./');

const fs = require('fs');
const {
	statSync,
	writeFileSync,
	readFileSync
} = require('fs');

const {
	dirname, 
	extname,
	basename,
	join, 
	normalize, 
	resolve, 
	relative,
	sep
} = require('path');

// 组装index.js，使用kbone模板
function makePageIndex() {
	const tmpl = `const mpRender = require('miniprogram-render');
const createPage = require('../base.js');
const global_config = require('../../config');
var avmPage = null;

function initPage($window, $document) {
	avmPage = require('./page.js')($window, $document);
}

// 事件兼容驼峰和纯小写
const LifeCycle = {
    onLoad(query){
		if(!avmPage){
			return;
		}
		if(!avmPage.$_notify(null, 'onLoad', query)){
			avmPage.$_notify(null, 'onload', query);
		}
    },
	onShow(){
		if(!avmPage){
			return;
		}
		if(!avmPage.$_notify(null, 'onShow')){
			avmPage.$_notify(null, 'onshow');
		}
    },
    onReady(){
		if(!avmPage){
			return;
		}
		if(!avmPage.$_notify(null, 'onReady')){
			avmPage.$_notify(null, 'onready');
		}
    },
    onHide(){
		if(!avmPage){
			return;
		}
		if(!avmPage.$_notify(null, 'onHide')){
			avmPage.$_notify(null, 'onhide');
		}
    },
	onUnload(){
		if(!avmPage){
			return;
		}
		if(!avmPage.$_notify(null, 'onUnload')){
			avmPage.$_notify(null, 'onunload');
		}
    },
	onPullDownRefresh(){
		if(!avmPage){
			return;
		}
		if(!avmPage.$_notify(null, 'onPullDownRefresh')){
			avmPage.$_notify(null, 'onpulldownrefresh');
		}
	},
	onReachBottom(){
		if(!avmPage){
			return;
		}
		if(!avmPage.$_notify(null, 'onReachBottom')){
			avmPage.$_notify(null, 'onreachbottom');
		}
	},
	onPageScroll(e){
		if(!avmPage){
			return;
		}
		if(!avmPage.$_notify(null, 'onPageScroll', e)){
			avmPage.$_notify(null, 'onpagescroll', e);
		}
	},
	onResize(size){
		if(!avmPage){
			return;
		}
		if(!avmPage.$_notify(null, 'onResize', size)){
			avmPage.$_notify(null, 'onresize', size);
		}
	},
	onTabItemTap(item){
		if(!avmPage){
			return;
		}
		if(!avmPage.$_notify(null, 'onTabItemTap', item)){
			avmPage.$_notify(null, 'ontabitemtap', item);
		}
	},
	onAddToFavorites(data){
		if(!avmPage){
			return;
		}
		return avmPage.$_notify(null, 'onAddToFavorites', data);
	},
	onShareAppMessage(data){
		if(!avmPage){
			return;
		}
		return avmPage.$_notify(null, 'onShareAppMessage', data);
	},
	onShareTimeline(){
		if(!avmPage){
			return;
		}
		return avmPage.$_notify(null, 'onShareTimeline');
	}
};

Page(createPage(mpRender, global_config, initPage, LifeCycle));`;
	return tmpl;
}

const globalVars = [
    'HTMLElement',
    'Element',
    'Node',
    'localStorage',
    'sessionStorage',
    'navigator',
    'history',
    'location',
    'performance',
    'Image',
    'CustomEvent',
    'Event',
    'requestAnimationFrame',
    'cancelAnimationFrame',
    'getComputedStyle',
    'XMLHttpRequest',
]

// 组装page.js，将编译的avm dsl代码按照kbone标准重构为为小程序cmj代码
function makePageJs(code) {
	// const bom = globalVars.map(item => `var ${item} = window.${item}`).join(';\n') + ';';
	// ${bom}
	const tmpl = `module.exports = function(window, document) {
	var apivm = require('../../common/avm.min')(window, document);
	var avm = apivm;
	var api = apivm.api;
	var Component = apivm.Component;
	var $ = apivm.$;
	${code}
	window['createApp'] = function() {
		apivm.$_start();
	}
	return avm;
}`;
	return tmpl;
}

/*
 组装index.json
 @pagePath stml文件路径，如果有同名的小程序json文件，则合并
 @option 与小程序json文件一一对于的传参，覆盖优先级高于stmlFilePath
*/
function makePageJson(pagePath, option) {
	var retJson = null; 
	if(pagePath){
		const jsonPath = pagePath.replace('.stml', '.json');
		if(pathExists(jsonPath)){
			retJson = JSON.parse(readFileSync(jsonPath).toString());
		}
	}
	const refreshEnable = option && option.enablePullDownRefresh ? true : false;
	if(!retJson){
		const tmpl = `{
			"enablePullDownRefresh": ${refreshEnable},
			"usingComponents": {
				"element": "miniprogram-element"
			}
		}`;
		retJson = JSON.parse(tmpl); 
	}
	if(!retJson.usingComponents){
		retJson.usingComponents = {};
	}
	retJson.usingComponents.element = "miniprogram-element";
	if(!retJson.hasOwnProperty("enablePullDownRefresh")){
		retJson.enablePullDownRefresh = refreshEnable;
	}
	return retJson;
}
/*
* 页面引入的外部css文件统一拷贝到common下
*/
function copyCssAsWxss(src, inputfile, outDir){
	const fname = basename(src, '.css');
	const csspath = resolve(dirname(inputfile), src);
	const wxsspath = join(outDir, `common/${fname}.wxss`);
	copyFileSync(csspath, wxsspath);
	return `@import "../../common/${fname}.wxss";\n`;
}

// 组装index.wxss
function makePageCss(result, inputfile, outDir) {
	if(!result.css) return false;
	let cssCode = result.css.code;
	const cssLink = result.css.link;
	if(cssLink.length > 0){
		cssLink.reverse();// 保持import顺序
		cssLink.forEach((src) => {
			const data = copyCssAsWxss(src, inputfile, outDir);
			cssCode = data + cssCode;
		});
	}
	return cssCode;
}

// 组装index.wxml
function makePageXml() {
	const tmpl = `<element 
		wx:if="{{pageId}}" 
		class="{{bodyClass}}" 
		style="{{bodyStyle}}" 
		data-private-node-id="e-body" 
		data-private-page-id="{{pageId}}" >
	</element>`;
	return tmpl;
}

exports.makePageIndex = makePageIndex;
exports.makePageJs = makePageJs;
exports.makePageJson = makePageJson;
exports.makePageCss = makePageCss;
exports.makePageXml = makePageXml;