/* Copyright 2020 APICloud Inc. All Rights Reserved.
 

 */

'use strict';

const fs = require('fs');
const statSync = fs.statSync;
const path = require('path');
const rollupPluginutils = require('rollup-pluginutils');

const { 
	warn,
	info
} = require('./client');


/*
* 只处理.stml文件
*/
function createFilter(include = [/\.stml$/i], exclude = []) {
    const filter = rollupPluginutils.createFilter(include, exclude);
    return id => filter(id);
}

function scan(folder) {
	if(!statSync(folder).isDirectory()){
		return null;
	}
	var lists = [];
	const issfc = createFilter();
	const paths = fs.readdirSync(folder);
	paths.forEach(function(pathname){
		let filedir = path.join(folder, pathname);
		let stat = statSync(filedir);
		if(stat.isFile()){
			if(issfc(filedir)){
				lists.push(filedir.toString());
			}
		}else if(stat.isDirectory()){
			// 不编译组件目录，组件只被编译到引用该组件的页面
			if(pathname.indexOf('components') >= 0){
				info('PreCompile: ignore "components" dir.');
				return;
			}
			var ls = scan(filedir);
			if(ls && ls.length > 0){
				lists = lists.concat(ls);
			}
		}
	});
	return lists;
}

// 扫描目录下的stml文件
function scanStmlSync(folder) {
	var lists = scan(folder);
	info("Scan stml total: " + (lists ? lists.length : '0'));
	return lists;
}

/*
* 同步复制文件夹
* @ src 源文件夹
* @ dst 目标文件夹
*/
function copyDirSync(src, dst) {
	//读取目录
	let paths = fs.readdirSync(src);
	paths.forEach(function(path){
		let _src = src + '/' + path;
		let _dst = dst + '/' + path;
		let _st = fs.statSync(_src);
		if(_st.isFile()){
			copyFileSync(_src, _dst);
		}else if(_st.isDirectory()){
			mkpath(_dst);
			copyDirSync(_src, _dst);
		}
	});
}
/*
* 同步复制文件
* @ src 源文件
* @ dst 目标文件
*/
function copyFileSync(src, dst) {
	mkpath(path.dirname(dst));
	//ensureParentDirsSync(dst);
	fs.writeFileSync(dst, fs.readFileSync(src));
}

/**
* 创建文件前，确保文件所在目录被同步创建
*/
function ensureParentDirsSync(dir) {
	if (fs.existsSync(dir)) {
		return
	}
	try {
		fs.mkdirSync(dir);
	} catch (err) {
		if (err.code === 'ENOENT') {
			ensureParentDirsSync(dirname(dir));
			ensureParentDirsSync(dir);
		}
	}
}

/**
* 同步创建目录
*/
function mkpath(dir) {
	if (fs.existsSync(dir)) {
		return true;
    } else {
		if (mkpath(path.dirname(dir))) {
			fs.mkdirSync(dir);
			return true;
		}
    }
}

/**
* 根据stml文件路径运算stml根路径
*/
function subStmlPath(filename) {
	if(filename){
		var indx = filename.indexOf('pages');
		if(indx >= 0){
			return filename.substr(0, indx);
		}
	}
	return filename;
}

function pathExists(path){
	try {
		fs.accessSync(path, fs.F_OK);
	} catch (e) {
		return false;
	}
	return true;
}

// no idea what these regular expressions do,
// but i extracted it from https://github.com/yahoo/js-module-formats/blob/master/index.js#L18
var ES6ImportExportRegExp = /(?:^\s*|[}{\(\);,\n]\s*)(import[\s+|{]['"]|(import|module)[\s+|*|{][^"'\(\)\n;]+[\s|}]+from\s*['"]|export[\s+|{](\*|\{|default|function|var|const|let|[_$a-zA-Z\xA0-\uFFFF][_$a-zA-Z0-9\xA0-\uFFFF]*))/;
var ES6AliasRegExp = /(?:^\s*|[}{\(\);,\n]\s*)(export\s*\*\s*from\s*(?:'([^']+)'|"([^"]+)"))/;

/**
* 是否是ES6模块，即通过export导出或通过import导入
*/
function isModule(sauce) {
  return ES6ImportExportRegExp.test(sauce)
    || ES6AliasRegExp.test(sauce);
};

module.exports = {
    scanStmlSync,
	subStmlPath,
    pathExists,
	mkpath,
	copyDirSync,
	copyFileSync,
	isModule
}