/* Copyright 2020 APICloud Inc. All Rights Reserved.

 */

const colors = require('colors');
const fs = require('fs')
const green = colors['green'];
const yellow = colors['yellow'];
const white = colors['white'];
const blue = colors['blue'];
const red = colors['red'];
const magenta = colors['magenta'];
const color = function(c){
	return colors[c];
}

// use for local debug
const local = false;

module.exports = {
	error,
	info,
	log,
	warn,
	green,
	yellow,
	white,
	blue,
	red,
	magenta,
	color,
	local,
	monitor,
	notifyStart,
	notifyEnd,
	state
}

var _initstamp;
var _laststamp;

function curtime(){
    return new Date().getTime();
}

function timestep(){
	const c = curtime();
	_laststamp = _laststamp || c;
	const t = c - _laststamp;
	_laststamp = c;
    return '  +' + t + 'ms';
}

function totaltime(){
	const c = curtime();
	_initstamp = _initstamp || c;
    return curtime() - _initstamp;
}

var _listener;

// 监听编译过程
/*
* listener:
* 	{
* 		onStart: function(msg){
* 		
* 		},
* 		onEnd: function(msg){
* 		
* 		},
* 		onConsole: function(msg){
* 		
* 		}
* 	}
* 
*/
function monitor(listener){
	_listener = listener;
}

// 编译开始
function notifyStart(){
	_initstamp = curtime();
	_laststamp = _initstamp;
	if(_listener && _listener.onStart){
		_listener.onStart();
	}
}

// 编译结束
function notifyEnd(stat){
	_end(stat);
	if(_listener && _listener.onEnd){
		_listener.onEnd(stat);
	}
}

// 编译过程输出
function _print(clor = white, msg, options = {}){
	const filename = options.filename;
	msg = `${msg}${filename ? ' at ' + filename : ''}`;
	msg += local ? green(timestep()) : timestep();
	msg = local ? clor(msg) : msg;
	if(_listener && _listener.onConsole){
		_listener.onConsole(msg);
	}else{
		console.log(msg);
	}
}

/**
	* print the error info by @babel/code-frame
	* @param codeFrame
	* @private
	*/
function _printCodeFrame(codeFrame){
	const { codeFrameColumns } = require('@babel/code-frame');
	const {e:{loc:start,message},filename} = codeFrame;
	if (!start) return;

	const res = codeFrameColumns(fs.readFileSync(filename).toString(), { start }, { highlightCode: true, message });
	const msg = red(`[ERROR] ❌ ---- ${filename}`)+` \n${res} \n ${red('----------------')}  `;

	if(_listener && _listener.onConsole){
		_listener.onConsole(msg);
	}else{
		console.log(msg);
	}
}

const DEFAULT_NAMESPACE = 'Tools.Todsl.';

function log(msg, namespace, options = {}){
	namespace = namespace || DEFAULT_NAMESPACE;
	msg = '[LOG] ' + namespace + msg;
	_print(white, msg, options);
}

function error(msg, namespace, options = {}){
	if (options.codeFrame){
		_printCodeFrame(options.codeFrame);
		msg = options.codeFrame.e.message;
	}
	namespace = namespace || DEFAULT_NAMESPACE;
	msg = '[ERROR] ' + namespace + msg;
	_print(red, msg, options);
}

function info(msg, namespace, options = {}){
	namespace = namespace || DEFAULT_NAMESPACE;
	msg = '[INFO] ' + namespace + msg;
	_print(green, msg, options);
}

function warn(msg, namespace, options = {}){
	namespace = namespace || DEFAULT_NAMESPACE;
	msg = '[WARN] ' + namespace + msg;
	_print(yellow, msg, options);
}

function debug(msg, options = {}){
	namespace = namespace || DEFAULT_NAMESPACE;
	msg = '[DEBUG] ' + namespace + msg;
	_print(blue, msg, options);
}

// BUILD in 42s. 26 TASKS: 26 SUCCEED, 0 FAILED
function _end(stat){
	let str = '';
	/*
	str += '[LOG] Finished.';
	str += ' -compiled ' + green(stat.total);
	str += ', -success ' + green(stat.ok.length);
	str += ', -aborted ' + red(stat.err.length);
	str += green(' +' + totaltime() + 'ms');
	*/
	str += '[LOG] Finished.';
	const t = (totaltime() / 1000).toFixed(2) + 's.';
	str += ' BUILD in ' + (local ? magenta(t) : t);
	str += ' ' + (local ? green(stat.total) : stat.total) + ' TASKS:';
	str += ' ' + (local ? green(stat.ok.length) : stat.ok.length) + ' SUCCEED';
	if(stat.err.length){
		str += ', ' + (local ? red(stat.err.length + ' FAILED') : stat.err.length + ' FAILED');
	}
	console.log(str);
}

function state(){
	const stat = {
		total: 0,
		ok: {
			length: 0,
			list: [],
			push: function(url){
				this.length++;
				this.list.push(url);
			}
		},
		err: {
			length: 0,
			list: [],
			push: function(url){
				this.length++;
				this.list.push(url);
			}
		},
		msg:''
	}
	return stat;
}