/* Copyright 2020 APICloud Inc. All Rights Reserved.


 */

function _interopDefault(ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }
const t = require('@babel/types');
const generate = _interopDefault(require('@babel/generator'));
const path = require('path');
const chalk = require('chalk');
const babelTraverse = _interopDefault(require('@babel/traverse'));
const babylon = _interopDefault(require('@babel/parser'));

const {
	info,
	error,
	log,
	warn,
	dev
} = require('../../tools/client');

const {
	autoCloseElements,
	elementMap,
	specialElement
} = require('./element-map')

/* 标准webComponents标签命名遵循"-"符连接规范，组件class遵循驼峰命名并且首字母大写 */
exports.normalizedComponentName = function normalizedComponentName (name) {
    name = name || 'sfc-compoennt';
	name = name.replace('_', '-').toLowerCase();
	const a = name.split('-').map(e => e[0].toUpperCase() + e.substr(1));
    return a.join('');
};

/* 自定义事件处理为标准DOM事件，命名遵循(on + 事件名 首字母大写)规范。例如：itemclick -> onItemclick */
exports.normalizedEventName = function normalizedEventName (name) {
    if(name){
		return 'on' + name.charAt(0).toUpperCase() + name.slice(1);
	}
    return name;
};

exports.basename = function basename (filename) {
    if (filename) {
		const basename = path.basename(filename);
        const a = basename.split('.');
		if(a && a.length >= 1){
			return a[0];
		}
    }
};

exports.log = log;
exports.logi = logi;
exports.error = error;
exports.warn = warn;
exports.info = info;

function logi(msg){
	log(msg);
}

function genAst (ast) {
	if(!ast){
		return '';
	}
    return generate(ast, {
		quotes: 'single',
		//retainLines: true
	}).code;
};

exports.generate = genAst;

exports.logast = logast;
function logast (msg, ast) {
	var str = '';
	if(Array.isArray(ast)){
		for(var i in ast){
			str += genAst(ast[i]) + ', ';
		}
	}else if(typeof ast === 'string'){
		str = ast;
	}else{
		str = genAst(ast);
	}
    info(msg + str);
};

exports.getIdentifier = getIdentifier;

// 根据key是在data中还是在props中，返回不同的结果。例如：key -> data.key | props.key | key
function getIdentifier (state, key) {
	// data为当前组件绑定数据，props为父组件绑定数据
	return state.data[key] ? t.identifier('data') : state.props[key] ? t.identifier('props') : t.identifier(key);
};

// 将export default中js的this.xxx转换为this.data.xxx或者this.props.xxx
exports.replaceThisExpression = function replaceThisExpression(path, state){
	const property = path.parent.property;
	if(!property){
		return;
	}
	const key = property.name;
	if('data' == key || 'props' == key){
		return;
	}
	if(propDefinite(state, key)){
		path.replaceWith(t.memberExpression(t.thisExpression(), getIdentifier(state, key)));
	}else if(computedsDefinite(state, key)){
		/* computed下的字段被挂载到了当前组件对象下，使用this.xxx即可引用，因此忽略
        path.parentPath.replaceWith(
            t.identifier(key)
        );
		*/
	}
	path.stop();
}

// 某字段是否包含在data或者props中定义
exports.propDefinite = propDefinite;
function propDefinite (state, key) {
	return state.data[key] || state.props[key];
};

// 某字段是否包含在computed中定义
exports.computedsDefinite = computedsDefinite;
function computedsDefinite (state, key) {
	return state.computeds[key];
};

// 某字段是否包含在computed或data或props中
exports.definiteInForAll = definiteInForAll;
function definiteInForAll (state, key) {
	return computedsDefinite(state, key) || propDefinite(state, key);
};

exports.testMemberExpression = testMemberExpression;
// 遍历<template>中某表达式，将其中在data/props/computed中定义的变量，转为this.data.xxx | this.props.xxx this.xxx 等，并返回表达式
// TODO 内部的step 2比较耗时，将来应做优化
function testMemberExpression(state, value){
	/* step 1 */
	let test = null;
	if(computedsDefinite(state, value)){
		// key在computed中
		test = t.memberExpression(t.thisExpression(), t.identifier(value));
	}else if(propDefinite(state, value)){
		// key在data或props中
		test = t.memberExpression(
			t.memberExpression(t.thisExpression(), getIdentifier(state, value)),
			t.identifier(value)
		);
	}
	/* step 2 */
	if(!test){
		const ast = babylon.parse(value);
		if (ast.program.body.length > 1) {
			throw new Error("Found multiple statements but wanted one");
		}
		if(t.isIdentifier(ast.program.body[0].expression)){
			// 只有一个字段的情况，已在step 1处理
			return null;
		}
		babelTraverse(ast, {
			Identifier(path){
				if('property' === path.key){
					// 以表达式"abc.xxx"为例，如果path的key为'property'，
					// 意味着Identifier对应的值为xxx，因此不需要处理，仅需处理其余类型（其余类型key可能为object、left、right、test等）
					// 可能需要处理 abc[this.data.xxx]类型，参照 directives.js > replaceItemInForChild
					return;
				}
				const name = path.node.name;
				const rts = testMemberExpression(state, name);
				if(rts){
					path.replaceWith(rts);
					// path.stop();
				}
			}
		});
		test = ast.program.body[0].expression;
	}
	return test;
}

// 获取FunctionExpression | ObjectMethod的参数
exports.getParams = function getParams(path, state){
	let params = [];
	if(t.isObjectMethod(path)){
		params = path.node.params; // add by moral.li 2019/10/24
	}else if(t.isObjectProperty(path)){
		const val = path.node.value;
		if(t.isFunctionExpression(val)){
			params = val.params;
		}
	}
	return params;
}

// 获取FunctionExpression | ObjectMethod的函数体，即花括号中内容
exports.getBody = function getBody(path, state){
	let body = path.node.body;
	if(t.isObjectMethod(path)){
		body = path.node.body;
	}else if(t.isObjectProperty(path)){
		const val = path.node.value;
		if(t.isFunctionExpression(val)){
			body = val.body;
		}
	}
	return body;
}

exports.genPropTypes = function genPropTypes (props) {
    const properties = [];
    const keys = Object.keys(props);

    for (let i = 0, l = keys.length; i < l; i++) {
        const key = keys[i];
        const obj = props[key];
        const identifier = t.identifier(key);

        let val = t.memberExpression(t.identifier('PropTypes'), t.identifier('any'));
        if (obj.type === 'typesOfArray' || obj.type === 'array') {
            if (obj.type === 'typesOfArray') {
                const elements = [];
                obj.value.forEach(val => {
                    elements.push(t.memberExpression(t.identifier('PropTypes'), t.identifier(val)));
                });
                val = t.callExpression(
                    t.memberExpression(t.identifier('PropTypes'), t.identifier('oneOfType')),
                    [t.arrayExpression(elements)]
                );
            } else {
                val = obj.required
                    ? t.memberExpression(t.memberExpression(t.identifier('PropTypes'), t.identifier('array')), t.identifier('isRequired'))
                    : t.memberExpression(t.identifier('PropTypes'), t.identifier('array'));
            }
        } else if (obj.validator) {
            const node = t.callExpression(
                t.memberExpression(t.identifier('PropTypes'), t.identifier('oneOf')),
                [t.arrayExpression(obj.validator.elements)]
            );
            if (obj.required) {
                val = t.memberExpression(
                    node,
                    t.identifier('isRequired')
                );
            } else {
                val = node;
            }
        } else {
            val = obj.required
                ? t.memberExpression(t.memberExpression(t.identifier('PropTypes'), t.identifier(obj.type)), t.identifier('isRequired'))
                : t.memberExpression(t.identifier('PropTypes'), t.identifier(obj.type));
        }

        properties.push(t.objectProperty(identifier, val));
    }

    // Babel does't support to create static class property???
    return t.classProperty(t.identifier('static propTypes'), t.objectExpression(properties), null, []);
};

exports.genDefaultProps = function genDefaultProps (props) {
    const properties = [];
    const keys = Object.keys(props).filter(key => typeof props[key].value !== 'undefined');

    for (let i = 0, l = keys.length; i < l; i++) {
        const key = keys[i];
        const obj = props[key];
        const identifier = t.identifier(key);

        let val = t.stringLiteral('error');
        if (obj.type === 'typesOfArray') {
            const type = typeof obj.defaultValue;
            if (type !== 'undefined') {
                const v = obj.defaultValue;
                val = type === 'number' ? t.numericLiteral(Number(v)) : type === 'string' ? t.stringLiteral(v) : t.booleanLiteral(v);
            } else {
                continue;
            }
        } else if (obj.type === 'array') {
            val = t.arrayExpression(obj.value.elements);
        } else if (obj.type === 'object') {
            val = t.objectExpression(obj.value.properties);
        } else {
            switch (obj.type) {
                case 'string':
                    val = t.stringLiteral(obj.value);
                    break;
                case 'boolean':
                    val = t.booleanLiteral(obj.value);
                    break;
                case 'number':
                    val = t.numericLiteral(Number(obj.value));
                    break;
            }
        }
        properties.push(t.objectProperty(identifier, val));
    }

    // Babel does't support to create static class property???
    return t.classProperty(t.identifier('static defaultProps'), t.objectExpression(properties), null, []);
};

const COLON_AND_AT = /(:\w+)|(@\w+)/g; // 匹配 :class :src @click @tap @click.native等

// 将Vue的':'冒号简写绑定数据，补全为v-bind:
// '@'符号简写绑定事件监听，补全为v-on:
function transformColonAndAt(code) {
	/*
	* @name 匹配到的字符串
	* @group 如果正则使用了分组匹配就为多个，否则无此参数
	* @rep 回调函数返回替换的值，如果没有返回，默认为undefined 
	* @offset 匹配字符串的对应索引位置
	* @allstr 原始字符串
	*/
    code = code.replace(COLON_AND_AT, (name, group, rep, offset, allstr) => {
        // const index = code.indexOf(name) - 1;
		const index = offset - 1;
		const last = code.charAt(index).trim();
		if(last){ // 如果上一个字符串不为空，说明不是:或@绑定，为空说明是 < :src='' @click>格式
			return name;
		}
		const first = name.charAt(0);
		if(':' === first){
			name = name.replace(':', 'v-bind:');
		}else if('@' === first){
			name = name.replace('@', 'v-on:');
		}
        return name;
    });
    return code;
}

// {{转换为{，}}转换为}
function transformBrace(code) {
    return code.replace(/{{/g, '{').replace(/}}/g, '}');
}

// 确保所有标签都是闭合的
function transformTagClosed(code) {
    if (!code) return '';

	var closed = function(tag, str){
		var t = `</${tag}>`;
		return str.endsWith('>') && (str.endsWith(t) || str.endsWith('/>'));
	}
	let tagReg
	// 自闭合标签补全，例如：<img src='xxx' > => <img src='xxx'></img>
    autoCloseElements.forEach(key => {
        tagReg = new RegExp("<" + key + "([\\s\\S]*?).*>", 'gim');
        code = code.replace(tagReg, (name, rep, offset, allstr) => {
			if(name && !closed(key, name)){
				warn(`JSX tag do not closing for <${key}> nearby: ` + name);
				//name = name + `</${key}>`;
				name = name.slice(0, name.length - 1) + '/>';
			}
			return name;
		});
    });
	/*
	// 自动结束标签规则优化：1. 未闭合判定完全置于正则中处理；2. 支持v-if中出现的大小写误判为结束问题。例如： <img v-if="a>1" class="a"> 是合法的未闭合
	autoCloseElements.forEach(key=>{
		code = code.replace(new RegExp(`<${key}(\\s+[^\\s<='"\\/>]+(=([^"'<\\/>]+|"[^"]*"|'[^']*'))?)*\\s*(?<!=)>(.*</${key}>)?`,'gim'),matchStr=>{
			if (matchStr.endsWith(`</${key}>`)){
				return  matchStr.replace(new RegExp(`\\s*></${key}>$`),' />')
			}else{
				warn(`JSX tag do not closing for <${key}> nearby: ` + matchStr);
				return  matchStr.replace(/\s*>$/,' />')
			}
		})
	})
*/
	/* // 特殊标签转换映射
	let target;
	Object.keys(elementMap).forEach(key => {
		target = elementMap[key];

	});
	*/
    return code;
}

// 清理js注释
function transformComments(code) {
	// replace目的是为了清除//注释？结果只清除了//，注释没被清除，导致babel/parser报错
    //return code.replace(/\/\//g, '');
	return code;
}

// 清理template注释
function transformTemplateComments(code) {
	return code.replace(/<!--[\w\W\r\n]*?-->/gmi, '');
}

// 预处理一些本编译器暂不支持的template写法
exports.htmlFaultTolerant = function htmlFaultTolerant(code){
	if(code){
		code = transformColonAndAt(code);
		code = transformBrace(code);
		code = transformTagClosed(code);
		code = transformTemplateComments(code);
	}
	return code;
}

// 预处理一些本编译器暂不支持的js写法
exports.jsFaultTolerant = function jsFaultTolerant(code){
	if(code){
		code = transformComments(code);
	}
	return code;
}

// 预处理一些本编译器暂不支持的css写法
exports.cssFaultTolerant = function cssFaultTolerant(code){
	if(code){
		code = code.trim(); //  //.replace(/^\s+/gm, '');
	}
	return code;
}
