/* Copyright 2020 APICloud Inc. All Rights Reserved.


 */

function _interopDefault(ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }
const t = require('@babel/types');
const babelTemplate = _interopDefault(require('@babel/template'));

//const cellMap = require('./cell-map');
const {
	isCell,
	isRecycler
} = require('./cell-map');
const { 
	log, 
	logast,
	warn,
	generate,
	testMemberExpression
} = require('../utils');

/*
*	查找v-if临近的下一个节点，判断其是否为else-if或else节点，并做相应操作
*/
exports.getNextIfJSXElement = getNextIfJSXElement;

function getNextIfJSXElement (path, state) {
	const nextElement = findNextConditionExpression(path, state);
	return nextElement;
};

let counterId = 0;
function conditionId() {
	counterId++;
    return 'c' + counterId;//o: object
}

/*
* 往v-if / elseif / else 节点插入 key 属性。vnode diff 过程有复用机制，
* 具有 key 关键字标识的节点，diff 算法将直接采用 key 进行运算，不复用
* @attrs ast节点的attributes
*/
function addKeyAttribute(attrs){
	if(!attrs){
		return;
	}
	for(let i = 0; i < attrs.length; ++i){
		const name = attrs[i].name.name;
		if(name == 'key'){ // 已定义key属性的节点不处理
			return;
		}
	}
	const name = t.jsxIdentifier("key");
	const jsxAttribute = t.jsxAttribute(name, t.stringLiteral(conditionId()));
	attrs.push(jsxAttribute);
}

/*
	在ast中，每个标签元素节点都属于其父节点的孩子，因此，path.key为孩子节点数组的下标，
	可通过该下标结合getSibling进行查找当前元素节点临近的上一个或下一个节点
*/
function findNextConditionExpression (path, state) {
    let nextElement = null;
    for (let i = path.key + 1; ; i++) {
        let nextPath = null;
		try{
			nextPath = path.getSibling(i);
		}catch (err) {
			//console.error(err);
		}
        if (!nextPath || !nextPath.node) {
            break;
        } else if (t.isJSXElement(nextPath.node)) {
			nextElement = nextPath.node;
			var hasElseIf = false;
			var hasElse = false;
			var hasCondition = false;
			const attrs = nextPath.node.openingElement.attributes;
			attrs.forEach(item => {
				const name = item.name.name;
				if(name === 'v-else-if'){
					hasElseIf = item.value.value;
					if(!hasElseIf){// 兼容jsx表达式：{doThat}
						hasElseIf = node.value;
					}
				}else if(name === 'v-else'){
					hasElse = true;
				}
			});
			hasCondition = hasElseIf || hasElse;
			// 相邻的下一个节点无else-if或else
			if(!hasCondition){
				nextElement = null;
				break;
			}
            nextPath.traverse({
                JSXAttribute (p) {
					const name = p.node.name.name;
                    if (name === 'v-else-if' || name === 'v-else') {
                        p.remove();
						return;
                    }
                }
            });
			// addKeyAttribute(attrs); // 先注释，目前未发现问题
			var expression = null;
			if(hasElseIf){
				// 有else-if，则递归组装表达式，直到else或者null
				// 返回值类似：condition ? element : element
				const test = testMember(state, hasElseIf);
				const next = findNextConditionExpression(nextPath, state);
				const consequent = nextElement;
				const alternate = next ? next : t.nullLiteral();
				expression = t.conditionalExpression(test, consequent, alternate);
			}
			if(nextElement){
				nextPath.remove();
			}
			if(expression){
				nextElement = expression;
			}
            break;
        }
    }
	// addKeyAttribute(path.node.openingElement.attributes); // 先注释，目前未发现问题
    return nextElement;
};

/*
	根据jsx表达式下的Identifier节点，查找其所在的整个jsx表达式节点
	例如：style={obj.item.xxx}，item为Identifier，则该函数返回{obj.item.xxx}节点	
*/
exports.getJSXExpressionContainer = function getJSXExpressionContainer (path) {
    let container = null;
	container = path.findParent(function(p){
		if(p.isJSXExpressionContainer()){
			return true;
		}
		return false;
	})
	return container;
};

// 是否是位于cell类型（可循环）标签内部
exports.isInCellJSXElment = function(path) {
    let nextElement = null;
	const cell = path.findParent(function(p){
		if(p.isJSXElement()){
			const name = p.node.openingElement.name.name;
			if(isCell(name)){
				return true;
			}
		}
		return false;
	});
	return cell ? true : false;
	// return cell;
};

// 是否是 cell 类型组件
exports.isCellJSXElment = function(path) {
	if(t.isJSXElement(path)){
		const name = path.openingElement.name.name;
		if(name && isCell(name)){
			return true;
		}
	}
	return false;
};

// 是否是可回收类型组件，例如 list-view 等
exports.isRecyclerJSXElment = function (path) {
	if(t.isJSXElement(path)){
		const name = path.node.openingElement.name.name;
		if(name && isRecycler(name)){
			return true;
		}
	}
	return false;
};

// 是否为事件类型的jsx属性，例如：onclick={this.func}
exports.isEventJSXAttribute = function isEventJSXAttribute (path) {
	const parent = path.parentPath;
	return parent && t.isJSXAttribute(parent.node) && parent.node.name.name.startsWith('on');
};

exports.genSFCRenderMethod = function genSFCRenderMethod (path, state, argument) {
	let blocks = [];
    /*
	const computedProps = Object.keys(state.computeds);
    if (computedProps.length) {
		// computed props
        computedProps.forEach(prop => {
            const v = state.computeds[prop];
            blocks = blocks.concat(v['_statements']);
        });
    }
	*/
    blocks = blocks.concat(t.returnStatement(argument));
    const render = t.classMethod(
        'method',
        t.identifier('render'),
        [],
        t.blockStatement(blocks)
    );

    path.node.body.push(render);
};

exports.isString = isString;

function isString(v){
	return v && ('string' === typeof v);
}

exports.testMember = testMember;

// 将指令值进行通配处理，转换为jsx中对应的有效js
function testMember(state, value){
	let test = null;
	if(isString(value)){
		// 处理: "doThat" 类型，处理为: this.data.doThat
		test = testMemberExpression(state, value);
	} else if(t.isJSXExpressionContainer(value)){
		// 处理：{doThat} 类型，处理为doThat
		test = value.expression;
	}
	if(!test){
		test = babelTemplate.expression.ast(value); // note: https://babeljs.io/docs/en/babel-template
	}
	return test;
}
