/* Copyright 2020 APICloud Inc. All Rights Reserved.


 */

/*
*
* 将 list-view grid-view 等组件内部可回收类型的 cell 等，转换为组件的 $bindCell_ 函数，用于native 调用
*/
/*
 In like:

<list-view cellTypeKey="type">
	<cell cellType="left" data-index={index}>
		<api-card v-bind:it={item}></api-card>
		<view>
			<image src={item.avatar}></image>
		</view>
	</cell>
	<cell cellType="right" data-index={index}>
		<api-card v-bind:it={item}></api-card>
		<view>
			<image src={item.avatar}></image>
		</view>
	</cell>
</list-view>

Out like:

<list-view cellTypeKey="type" $bindCell_ = function(celltype, item, index){
	return celltype == 'left' ? <cell cellType="left" data-index={index}>
			<api-card v-bind:it={item}></api-card>
			<view>
				<image src={item.avatar}></image>
			</view>
	</cell> : celltype == 'right' ? <cell cellType="right" data-index={index}>
			<api-card v-bind:it={item}></api-card>
			<view>
				<image src={item.avatar}></image>
			</view>
		</cell> : null;
}>

</list-view>

*/

function _interopDefault(ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }
const t = require('@babel/types');
const babelTemplate = _interopDefault(require('@babel/template'));
const babelTraverse = _interopDefault(require('@babel/traverse'));
const babylon = _interopDefault(require('@babel/parser'));

const {
	isCellJSXElment
} = require('./sfc-ast-helpers');
const { 
	log,
	logi,
	logast, 
	warn,
	error
} = require('../utils');

function getCellType(attrs){
	if(!attrs || attrs.length <= 0){
		return;
	}
	let value;
	for(let i = 0; i < attrs.length; ++i){
		const name = attrs[i].name.name;
		if(name == 'cellType' || name == 'cell-type'){
			value = attrs[i].value.value;
			break;
		}
	}
	return value;
}

/*
* 是否有v-for指令，是则不处理。swiper支持v-for。
*/
function hasFor(attrs){
	if(!attrs || attrs.length <= 0){
		return false;
	}
	for(let i = 0; i < attrs.length; ++i){
		const name = attrs[i].name.name;
		if(name == 'v-for'){
			return true;
		}
	}
	return false;
}

/*
* 是否有 reusable 属性，只有 reusable 标识的 swiper-item 才应被编译为可回收。
*/
function isNotReusableItem(name, attrs){
	if(name == 'swiper-item'){
		for(let i = 0; i < attrs.length; ++i){
			const name = attrs[i].name.name;
			if(name == 'reusable'){
				return false;
			}
		}
		return true;
	}
	return false;
}

function makeConditionalExpression(cells){
	let expression;
	for (let i = 0; i < cells.length; i++) {
		const obj = cells[i];
		const _type = obj.key;
		const _cell = obj.value;
		
		const val = `celltype == "${_type}"`;
		const test = babelTemplate.expression.ast(val);
		const next = expression;
		const consequent = _cell;
		const alternate = next ? next : t.nullLiteral();
		expression = t.conditionalExpression(test, consequent, alternate);
	}
	//logast('makeConditionalExpression >>>: \n', expression);
	return expression;
}

const DEFAULT_CELL_TYPE = "cell_def";

module.exports = function (path, state) {
	logi('handleRecyclerElement[1]   ');
	
	const children = path.node.children;
	if(!children || children.length <= 0){
		return;
	}
	const cells = []; // cell 类型子节点
	let hasCellType = false;
	children.forEach((child, index) => {
		if(!isCellJSXElment(child)){
			return;
		}
		const attrs = child.openingElement.attributes;
		if(hasFor(attrs)){
			return;
		}
		const name = child.openingElement.name.name;
		if(isNotReusableItem(name, attrs)){
			return;
		}
		let cellType = getCellType(attrs);
		if(!cellType){
			cellType = DEFAULT_CELL_TYPE;
		}else{
			hasCellType = true;
		}
		cells.push({
			key: cellType,
			value: child
		});
		delete children[index];
	});
	if(0 === cells.length){
		return;
	}
	let cellExpression;
	if(!hasCellType || 1 == cells.length){
		cellExpression = cells[0].value;
	}else{
		cellExpression = makeConditionalExpression(cells);
	}
	
	const name = t.jsxIdentifier('$bindCell_');
	const params = [t.identifier('celltype'), t.identifier('item'), t.identifier('index')];
	const jsxExpressionContainer = t.jsxExpressionContainer(
		t.arrowFunctionExpression(
			params,
			t.blockStatement([
				t.returnStatement(cellExpression)
			])
		));
	const jsxAttribute = t.jsxAttribute(name, jsxExpressionContainer);
	path.node.openingElement.attributes.push(jsxAttribute);
	
	logi('handleRecyclerElement[2]   ');
};

