/* Copyright 2020 APICloud Inc. All Rights Reserved.


 */

function _interopDefault(ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

const babylon = require('@babel/parser');
const t = require('@babel/types');
const babelTraverse = require('@babel/traverse').default;
const babelTemplate = _interopDefault(require('@babel/template'));

const { 
	logi, 
	logast, 
	warn, 
	error, 
	getIdentifier,
	computedsDefinite,
	propDefinite,
	definiteInForAll,
	generate,
	testMemberExpression
} = require('../utils');
const { 
    handleIfDirective, 
	handleShowDirective, 
	handleOnDirective,
    handleForDirective, 
	handleTextDirective, 
	handleHTMLDirective,
    handleBindDirective,
    handleModelDirective,
    handleOnEvent
} = require('./directives');
const handleRecyclerElement = require('./recycler');
const { 
	isInCellJSXElment,
	isRecyclerJSXElment,
	isEventJSXAttribute,
	getJSXExpressionContainer
} = require('./sfc-ast-helpers');


/*
	解析处理template标签内容
*/
module.exports = function traverseTemplate (template, state) {

	logi('TraverseTemplate[1]  ');

	let argument = null;
    // cache some variables are defined in v-for directive
    const definedInFor = [];
    // AST for template in sfc
    const tast = babylon.parse(template, {
        sourceType: 'module',
        plugins: ['jsx']
    });
	
	/* 
		step 1 
		处理vue指令
	*/
    babelTraverse(tast, {
        
		ExpressionStatement: {
            enter (path) {
                
            },
            exit (path) {
                argument = path.node.expression;
            }
        },

		JSXElement (path){
			if(isRecyclerJSXElment(path)){
				handleRecyclerElement(path, state);
			}
		},

        JSXAttribute (path) {
            const node = path.node;
			// 类似于tapmode percent之类的属性，可能没有value，或者value为null
			// 不处理，否则将异常 2019/12/03 by moral.li
			if (!node.name || !node.value) {
                return;
            }
            var value = node.value.value;
			if(!value){// v-*指令一般为字符串"doThat"，兼容jsx表达式{doThat}则通过node.value.value获取值
				value = node.value;
			}
			var name = node.name.name.toString();
            if (name === 'v-show') {
				try{
					handleShowDirective(path, value, state);
				}catch (err) {
					error('handle v-show directive error: ' + err);
				}
            } else if (name === 'v-for') {
                try{
					handleForDirective(path, value, definedInFor, state);
				}catch (err) {
					error('handle v-for directive error: ' + err);
				}
            } else if (name === 'v-text') {
                try{
					handleTextDirective(path, value, state);
					path.remove();
				}catch (err) {
					error('handle v-text directive error: ' + err);
				}
            } else if (name === 'v-html') {
				try{
					handleHTMLDirective(path, value, state);
				}catch (err) {
					error('handle v-html directive error: ' + err);
				}
            } else if(name.startsWith('on')){
            	//处理原生H5标签写法的事件 <view onclick="onClick()"></view>
            	try{
            		handleOnEvent(path);
				}catch (err) {
					error('handle onEvent directive error: ' + err);
				}
            } else if (t.isJSXNamespacedName(node.name)) {
                try{
					if (node.name.namespace.name === 'v-on') {
						name = node.name.name.name;
						handleOnDirective(path, name, value);
					} else if (node.name.namespace.name === 'v-bind') {
						name = node.name.name.name;
						handleBindDirective(path, name, value, state);
					}
				}catch (err) {
					error('handle v-on or v-bind directive error: ' + err);
				}
            }

			/*
			if (name === 'class' || name === 'scrollY') {
                // TODO 也可在此做一些转换或者兼容，例如scrollY -> scroll-y
            }
			*/
        }
    });

	logi('TraverseTemplate[2]  ');
	/*
		step 2 
		处理v-if及v-model指令，同时补全表达式中的字段引用为this.data.xxx | this.props.xxx等
		v-for优先级比v-if高，因此在step 1中先处理v-for，
		防止v-if下第一个节点是v-for时，因编译被replaceWith污染而导致没有container引起失败
	*/
	babelTraverse(tast, {
        
		ExpressionStatement: {
            enter (path) {
                
            },
            exit (path) {
                argument = path.node.expression;
            }
        },

        JSXAttribute (path) {
			const node = path.node;
			if (!node.name || !node.value) {
                return;
            }
            var value = node.value.value;
			if(!value){// 兼容jsx表达式：{doThat}
				value = node.value;
			}
            if (node.name.name === 'v-if') {
				try{
					handleIfDirective(path, value, state);
				}catch (err) {
					error('handle v-if directive error: ' + err);
				}
            }  else if(node.name.name === 'v-model'){
            	try{
					handleModelDirective(path, value);
				}catch (err) {
					error('handle v-model directive error: ' + err);
				}
            }
		},
		
		// 补全处理jsx表达式中引用的data/computeds/props值。
		// 并处理专门为原生APP定制的cell类型数据遍历回调关键字'item'为'$bind'前缀的字符串
        JSXExpressionContainer (path) {
			if(isEventJSXAttribute(path)){
				// 事件对应的表达式不处理
				return;
			}
			const elmentInCells = isInCellJSXElment(path);
			path.traverse({
				Identifier (p) {
					const name = p.node.name;
					if('property' === p.key){
						// 以表达式"abc.xxx"为例，如果path的key为'property'，
						// 意味着Identifier对应的值为xxx，因此不需要处理，仅需处理其余类型（其余类型key可能为object、left、right、test等）
						return;
					}
					if(('item' === name || 'index' === name) && elmentInCells){
						// 只对包含item字段且位于cell节点下的表达式进行'$bind.'处理
						/* 对于list-view等，已通过将cell编译为$bindCell_函数进行处理，此处注销
						var container = getJSXExpressionContainer(p);
						if(!container){
							return;
						}
						var value = generate(container.node.expression);
						value = '$bind.' + value;
						const parent = container.parentPath.node;
						if(t.isJSXElement(parent)){
							value = t.jsxText(value);
						}else if(t.isJSXAttribute(parent)){
							value = t.stringLiteral(value);
						}else{
							value = t.stringLiteral(value);
						}
						container.replaceWith(value);
						*/
					}else{
						// 其余做data/computeds/props补全
						if(definedInFor.includes(name) || !p.container){
							return;
						}
						let test = testMemberExpression(state, name);
						if(test){
							p.replaceWith(test);
						}
					}
				}
			});
        }
	});

	logi('TraverseTemplate[3]  ');
    return argument; 
};
