/* Copyright 2020 APICloud Inc. All Rights Reserved.


 */

/*
*
* 转换Vue指令: v-if v-for v-bind等 
*/

// jsx 事件属性绑定语法规则：https://blog.csdn.net/weixin_34194551/article/details/91367520

function _interopDefault(ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }
const t = require('@babel/types');
const babelTemplate = _interopDefault(require('@babel/template'));
const babelTraverse = _interopDefault(require('@babel/traverse'));
const babylon = _interopDefault(require('@babel/parser'));

const {
	isString,
	testMember,
	getNextIfJSXElement,
	parseArgs
} = require('./sfc-ast-helpers');
const { 
	log, 
	logast, 
	warn,
	error, 
	getIdentifier, 
	propDefinite, 
	computedsDefinite, 
	definiteInForAll, 
	normalizedEventName,
	testMemberExpression,
	generate
} = require('../utils');
const eventMap = require('./event-map');


let emptyIder = 0;
function emptyId() {
	emptyIder++;
    return 'e' + emptyIder;
}

const useElement = false;

// 构建一个空的占位节点. 返回类似：<view />
function emptyJSXElement(){
	if(!useElement){
		return t.nullLiteral();
	}
	const key = t.jsxIdentifier("key");
	const jsxAttribute = t.jsxAttribute(key, t.stringLiteral(emptyId()));

	const name = t.jsxIdentifier("view");
	const openingElement = t.jsxOpeningElement(name, [], true);
	const closingElement = null;
	const children = [];
	const selfClosing = true;
	return t.jsxElement(openingElement, closingElement, children, selfClosing);
}

// 转换v-if v-else-if v-else
exports.handleIfDirective = function handleIfDirective (path, value, state) {
	const parentPath = path.parentPath.parentPath;
	if(!parentPath.container){
		warn('no container! with handle v-if directive near \'' + value + '\'');
		return;
	}
    const childs = parentPath.node.children;
	const test = testMember(state, value);
	// Get JSXElement of v-else-if v-else or none
    const nextElement = getNextIfJSXElement(parentPath, state);
	const consequent = parentPath.node;
	const alternate = nextElement ? nextElement : emptyJSXElement(); // t.nullLiteral()
	const ts = t.jSXExpressionContainer(t.conditionalExpression(test, consequent, alternate));
    parentPath.replaceWith(ts);
    path.remove();
};

// 转换v-show
exports.handleShowDirective = function handleShowDirective (path, value, state) {
	const test = testMember(state, value);
    path.replaceWith(
        t.jSXAttribute(
            t.jSXIdentifier('style'),
            t.jSXExpressionContainer(
                t.objectExpression([
                    t.objectProperty(
                        t.identifier('display'),
                        t.conditionalExpression(
                            test,
                            t.stringLiteral('flex'),
                            t.stringLiteral('none')
                        )
                    )
                ])
            )
        )
    );
};

// 转换v-on
exports.handleOnDirective = function handleOnDirective (path, name, value) {
	let eventName = eventMap[name];
    if (!eventName) {
		warn(`a customized event name: ` + name);
		eventName = normalizedEventName(name);
    }
	// 处理 类h5类型
	if(isString(value)){
		// 处理: v-on:click="doThat('hello')" 类型，处理为jsx: onClick={() => this.doThat('hello')}
		if(value.indexOf('(') != -1 && value.indexOf(')') != -1) {
			const newParams = [];
			path.replaceWith(
				t.jSXAttribute(
					t.jSXIdentifier(eventName),
					t.jSXExpressionContainer(
						t.arrowFunctionExpression(
							newParams,
							t.memberExpression(
								t.thisExpression(),
								t.identifier(value)
							)
						)
					)
				)
			);
		}else{
			// 处理：v-on:click="doThat" 类型，处理为: onClick={this.doThat}
			path.replaceWith(
				t.jSXAttribute(
					t.jSXIdentifier(eventName),
					t.jSXExpressionContainer(
						t.memberExpression(
							t.thisExpression(),
							t.identifier(value)
						)
					)
				)
			);
		}
		return;
	}
	// 处理：v-on:click={this.doThat} 类型（jsx类型）
	if(t.isJSXExpressionContainer(value)){
		path.replaceWith(
			t.jSXAttribute(
				t.jSXIdentifier(eventName),
				value
			)
		);
	}
};

// 转换v-bind
exports.handleBindDirective = function handleBindDirective (path, name, value, state) {
	const test = testMember(state, value);
	/*
	if('class' === name){
		// 支持vue的将静态class与动态绑定class区分
		name = 'classLive';
	}
	*/
	path.replaceWith(
        t.jSXAttribute(
			t.jSXIdentifier(name),
			t.jSXExpressionContainer(test)
		)
    );
};

const ITEM = 'item';
const $ITEM = 'item$1';
const INDEX = 'index';
const $INDEX = 'index$1';

// "(arg1, arg2) in | of arrValue" to ['(arg1, arg2)', 'arrValue']
function parseForValue(value){
	// vue支持in和of遍历
	var val = value.split(/\s+?in\s+?/);
	if(!val || val.length <= 1){
		val = value.split(/\s+?of\s+?/);
	}
	return val;
}

// parse "(arg1, arg2, argn)" to ['arg1', 'arg2', 'argn']
function parseForArgs(value){
	return value.replace(/\s*/g,"").replace('(', '').replace(')', '').split(',');
}

// 转换v-for
exports.handleForDirective = function handleForDirective (path, value, definedInFor, state) {
	var parentPath = path.parentPath.parentPath;
	if(!parentPath.container){
		warn('no container! with handle v-for directive near \'' + value + '\'');
	}
	// 需处理最外层的其他属性，例如class等 path.node.attributes
    const childs = parentPath.node.children;
    const element = parentPath.node.openingElement.name.name;
	const attrs = parentPath.node.openingElement.attributes;
    const test = parseForValue(value);
	if(!test || test.length <= 1){
		error(` A error directive value: ${value}`);
		path.remove(); // 如果Vue v-for指令写法不正确，则忽略该节点
		return;
	}
    const condition = test[1].trim();
    const params = parseForArgs(test[0]);
    const newParams = [];
	var hasIndexKey = false;
    params.forEach(param => {
		const p = param;
		var $param = p;
		// 将以'item'做为遍历key进行重命名，防止与list-view特定的'item'遍历值冲突
		// 见./index.js -> // step 2 -> JSXExpressionContainer
		if(ITEM === $param){
			$param = $ITEM;
			replaceChildQuoted(parentPath, ITEM, $param);
		}else if(INDEX === $param){
			$param = $INDEX;
			replaceChildQuoted(parentPath, INDEX, $param);
		}
        definedInFor.push($param);
        newParams.push(t.identifier($param));
		if(INDEX === p){
			hasIndexKey = true;
		}
    });
	const newAttrs = [];
	attrs.forEach(item => { // 除v-for外属性要还原回节点去
        if('v-for' !== item.name.name){
			newAttrs.push(item);
		}
    });
	/*
	if(hasIndexKey){ // 如果v-for中带有index索引，则绑定索引到key
		newAttrs.push(t.jSXAttribute(
			t.jSXIdentifier('key'),
			t.jSXExpressionContainer(
				t.identifier('index')
			)
		));
	}
	*/

	const arrayExpression = genForEachMemberExpression(state, condition);
	const returnStatement = t.returnStatement(
		t.jSXElement(
			t.jSXOpeningElement(t.jSXIdentifier(element), newAttrs),
			t.jSXClosingElement(t.jSXIdentifier(element)),
			childs
		)
	);

	const arrowFunction = t.arrowFunctionExpression(
		newParams,
		t.blockStatement([returnStatement])
	);

    parentPath.replaceWith(
        t.jSXExpressionContainer(
            t.callExpression(
				arrayExpression,
                [
                    arrowFunction
                ]
            )
        )
    );

};

// 转换v-text
exports.handleTextDirective = function handleTextDirective (path, value, state) {
    const parentPath = path.parentPath.parentPath;
	const test = testMember(state, value);
	parentPath.node.children.push(
		t.jSXExpressionContainer(
			t.callExpression(
				t.memberExpression(test,
					t.identifier('replace')
				),
				[
					t.regExpLiteral('<[^>]+>', 'g'),
					t.stringLiteral('')
				]
			)
		)
	);
};

// 转换v-html
exports.handleHTMLDirective = function handleHTMLDirective (path, value, state) {
    const val = state.computeds[value] ? t.identifier(value) : t.memberExpression(
        t.memberExpression(t.thisExpression(), getIdentifier(state, value)),
        t.identifier(value)
    );

    path.replaceWith(
        t.jSXAttribute(
            t.jSXIdentifier('dangerouslySetInnerHTML'),
            t.jSXExpressionContainer(
                t.objectExpression(
                    [
                        t.objectProperty(t.identifier('__html'), val)
                    ]
                )
            )
        )
    )
};

//转换v-model
exports.handleModelDirective = function handleModelDirective (path, value, state) {    
    var onInputPath = null;
    for(var i = 0; i < path.container.length; i++){
        var nextPath = path.getSibling(i);
        if(nextPath.node.name.name.toLowerCase() === 'oninput' ){
            onInputPath = nextPath;
        }
        if(nextPath.node.name.name === 'value'){
            nextPath.remove();
        }
    }

    var newInputFun = '';
    if(onInputPath){
        newInputFun = generate(onInputPath.node.value.expression);
        if(onInputPath.node.value.expression.type == 'ArrowFunctionExpression'){
            newInputFun = `(${newInputFun})(e);`;
        }else if(onInputPath.node.value.expression.type == 'MemberExpression'){
            newInputFun = `${newInputFun}(e);`;
        }
        onInputPath.remove();
    }
	var leftName = '';
    if(typeof(value) === 'string'){
    	leftName = value.split('.')[0];
    }else{
	    leftName = value.expression.name;
		if(t.isMemberExpression(value.expression)){
			if (t.isMemberExpression(value.expression)) {
		        if (!t.isThisExpression(value.expression.object)) {
		            leftName = value.expression.object.name
		        }
		    }
		}
    }

    value = (typeof(value) === 'string' ? value : value.expression.name);
    const ast = babylon.parse('(e)=>{' + 'if(typeof ' + leftName + ' != "undefined"){' + value + ' = e.target.value; }else{ this.data.' + value + ' = e.target.value;} '  + newInputFun + '}');

    path.replaceWith(
        t.jSXAttribute(
            t.jSXIdentifier('onInput'),
            t.jSXExpressionContainer(
                ast.program.body[0].expression
            )
        )
    );

    const ast2 = babylon.parse('typeof ' + leftName + ' == "undefined" ? this.data.' + value + ':' + value);
    var valueAttrNode = t.jSXAttribute(
        t.jSXIdentifier('value'),
        t.jSXExpressionContainer(
            ast2.program.body[0].expression
        )
    );
    path.parentPath.node.attributes.push(valueAttrNode)
};

exports.handleOnEvent = function handleOnEvent (path) {
    // TODO 未处理响应函数中 this 指针
	if(t.isStringLiteral(path.node.value)){
        var value = path.node.value.value;
        if(value.indexOf('(') === -1){
            value += '(event);'
        }
        var funName = value.split('(')[0].split('.')[0];
        var str = '(event)=>{'+
            'if(this.' + funName + '){' +
                'this.' + value + 
            '}else{' +
                value +
            '}' +
        '}';
        const ast = babylon.parse(str);
        path.replaceWith(
            t.jSXAttribute(
                t.jSXIdentifier(path.node.name.name),
                t.jSXExpressionContainer(
                    ast.program.body[0].expression
                )
            )
        );
    }
}

function replaceIdentifier(path, target, replacement){
	if('property' === path.key){
		const parentPath = path.parentPath;
		if(parentPath && !parentPath.node.computed){
			// 类似xxx.item终止，而xxx[item]则继续
			return;
		}
	}
	var name = path.node.name;
	if(target === name){
		path.replaceWith(t.identifier(replacement));
		// path.stop(); // stop会引起遍历中断，导致类似{item ? item.xxx : 'xxx'}的第二个item被忽略
	}
}

/* 
 如果v-for的key使用了item关键字定义，则将item进行重命名，
 防止与list-view定义的item冲突（list-view具备cell回收能力的组件内部做了特殊处理，cell通过item关键字接收来自原生app的数据回调）.
 @ inforPath v-for指令所在节点
 @ target 替换目标
 @ replacement 替换值
*/
function replaceChildQuoted(inforPath, target, replacement){
	inforPath.traverse({
		// 处理通过表达式引用的v-for条件，例如：<text>{item.xxx}</text>
		Identifier (path) {
			replaceIdentifier(path, target, replacement);
		},
		// 处理通过字符串引用的v-*指令，例如：<text v-if="item.xxx == 2"></text>
		JSXAttribute (path){
			const node = path.node;
			if (!node.name || !node.value) {
                return;
            }
			var name = node.name.name.toString();
			if(t.isJSXNamespacedName(node.name)){
				name = node.name.namespace.name;
			}
			if(!name.startsWith('v-')){
				return;
			}
			var vnode = node.value;
			if(vnode && t.isStringLiteral(vnode)){
				if(vnode.value.indexOf(target) < 0){
					return;
				}
				// 通过字符串全局替换的方式可能会造成误判，改进为使用 ast 进行遍历更新
				try{
					const val = vnode.value.replace(/[\t\r\f\n]*/g, '');
					const ast = babylon.parse(val);
					babelTraverse(ast, {
						Identifier(path){
							replaceIdentifier(path, target, replacement);
						}
					});
					const str = generate(ast.program.body[0].expression);
					if(str){
						vnode.value = str;
						return;
					}
				}catch (err) {
					;
				}
				const re =new RegExp(target, 'g');
				vnode.value = vnode.value.replace(re, replacement);
			}
		}
	});
	return inforPath;
}

// 根据v-for渲染的对象是数组还是对象区分返回不同遍历
function genForEachMemberExpression(state, prop) {
	let member = '';
	// testMemberExpression内部将任意数据格式根据其所挂载在data|props|computed，
	// 处理为为this.data.xxx | this.props.xxx | this.xxx等
	// 兼容dataMap || dataMap.item_1 || dataMap['item_1'] || dataMap[item_1]等方式访问
	let test = testMemberExpression(state, prop);
	if (test) {
		member = generate(test);
	} else {
		member = prop;
		//test = babelTemplate.expression.ast(prop);// note: https://babeljs.io/docs/en/babel-template
	}
	const memberStr = `(Array.isArray(${member}) ? ${member} : Object.values(${member})).map`;
	member = babylon.parse(memberStr).program.body[0].expression;
	return member;
}
    // 数据格式多样性，要分别处理，如
    // dataMap || dataMap.item_1 || dataMap['item_1'] || dataMap[item_1] (此处的item_1需要另外检查，是否在data或computed中)
    // 转换成，如果在data中如下
    // this.data.dataMap || this.data.dataMap.item_1 || this.data.dataMap['item_1'] || this.data.dataMap[this.data.item_1](此处的item_1需要另外检查)
    // 穿换成，如果在computed中如下
    // this.dataMap || this.dataMap.item_1 || this.dataMap['item_1'] || this.dataMap[this.item_1](此处的item_1需要另外检查)
