/* Copyright 2020 APICloud Inc. All Rights Reserved.


 */

/**
* Babel AST 类型文档：https://github.com/babel/babylon/blob/master/ast/spec.md 
* Babel AST 访问API文档：https://github.com/jamiebuilds/babel-handbook/blob/master/translations/en/plugin-handbook.md
* Babel AST 访问API文档：https://www.babeljs.cn/docs/babel-types
* Babel在7.0后改名为@Babel
*/
/* babel遍历回调 path字段：

	{
		"node": 节点
		"parent": 父节点
		"parentPath": 父path
		"scope": 作用域
		"key": 当前节点在ast中的访问关键字，例如object | property | left | right等,
		"type": 当前节点类型，例如MemberExpression | Identifier | JSXElement等,

		"hub": {...},
		"contexts": [],
		"data": {},
		"shouldSkip": false,
		"shouldStop": false,
		"removed": false,
		"state": null,
		"opts": null,
		"skipKeys": null,
		"context": null,
		"container": null,
		"listKey": null,
		"inList": false,
		"parentKey": null,
		"typeAnnotation": null
	}

*/

/**
* 
* ast预览地址：https://astexplorer.net/ 选择语言类型为JavaScript，解析工具为@babel/parser
*
*/

function _interopDefault(ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

const fs = require('fs');
const {
	dirname, 
	extname, 
	join, 
	normalize, 
	resolve, 
	relative,
	sep
} = require('path');
const babylon = require('@babel/parser');
const babelTraverse = _interopDefault(require('@babel/traverse'));
const generate = _interopDefault(require('@babel/generator'));
const t = require('@babel/types');
const vuecompiler = require('vue-template-compiler');

const { 
	css2Json,
	saveAsJson
} = require('./collect-styles');
const { 
	basename, 
	normalizedComponentName,
	info, 
	logi, 
	warn, 
	error, 
	logast,
	htmlFaultTolerant, 
	jsFaultTolerant, 
	cssFaultTolerant
} = require('./utils');

const { 
	collectProps, 
	collectData, 
	collectComputed, 
	collectComponents 
} = require('./collect-state');
const { collectDeclaration } = require('./collect-declarations');
const { 
    collectCycleMethods, 
    collectGeneralMethods,
	isGeneralMethods
} = require('./vue-ast-helpers');

const { 
    genImports, 
	genConstructor,
    genStaticProps, 
	genClassMethods
} = require('./react-ast-helpers');
const { genDeclarations } = require('./declarations-ast-helpers');
const { genSFCRenderMethod } = require('./sfc/sfc-ast-helpers');

const bublejsx = require('./buble-jsx');
const format = require('./output');
const traverseTemplate = require('./sfc/index');
/* Vue生命周期映射为apivm生命周期 */
const cycle = require('./cycle-map');

const collect = { 
    imports: [],
    classMethods: {},
	declarations: [],
	reset: function(){
		this.imports = [];
		this.classMethods = {};
		this.declarations = [];
	}
};

const state = {
    name: undefined,
    data: {},
    props: {},
    computeds: {},
    components: {},
	reset: function(){
		this.name = undefined;
		this.data = {};
		this.props = {};
		this.computeds = {};
		this.components = {};
	}
};

function formatContent (source, isSFC) {
    if (isSFC) {
        const res = vuecompiler.parseComponent(source, { pad: 'space' });
		const t = res.template ? htmlFaultTolerant(res.template.content) : '';
		const j = res.script ? jsFaultTolerant(res.script.content) : '';
		const s = res.styles ? res.styles : null;
        return {
            template: t,
            js: j,
			styles: s
        };
    } else {
        return {
            template: null,
            js: source,
			styles: null
        };
    }
}

// AST for vue of avm component
function transformForComponent(source, filename, isSFC, isImport) {

	logi('Transpile: enter     ');

	collect.reset();
	state.reset();

	const component = formatContent(source.toString(), isSFC);

	logi('FormatContent[1]     ');
	
	/* ----------------------------- 开始收集SFC中数据，暂存到state, collect ------------------------------ */

    const vast = babylon.parse(component.js, {
        sourceType: 'module',
        plugins: isSFC ? [] : ['jsx','classProperties']
    });
	
	logi('CollectState[1]      ');

    collectProps(vast, state);			// 收集SFC中VUE Component的Props暂存到state
    collectData(vast, state);			// 收集SFC中VUE Component的Data暂存到state
    collectComputed(vast, state);		// 收集SFC中VUE Component的computed暂存到state
    collectComponents(vast, state);		// 收集SFC中VUE Component的components暂存到state
	collectDeclaration(vast, collect);	// 收集SFC中在Export Default以外区域定义的函数/对象等暂存到collect

	logi('CollectState[2]      ');

	// 收集SFC中定义的import、methods以及生命周期函数暂存到collect
    babelTraverse(vast, {

        ImportDeclaration (path) {
            collect.imports.push(path.node);
        },
    
        ObjectMethod (path) {
            const name = path.node.key.name;
            if (isGeneralMethods(path)) {
				// 收集通过 key(){ } 方式写的Vue methods函数
				collectGeneralMethods(path, collect, state, name);
            } else if (cycle[name]) {
				// 收集Vue的生命周期函数
                collectCycleMethods(path, collect, state, name, cycle[name], isSFC);
            } else {
                if (name === 'data' || state.computeds[name]) {
                    return;
                }
                warn(`The ${name} method maybe be not support now`);
				// 暂未支持的生命周期函数编译
				collectCycleMethods(path, collect, state, name, name, isSFC);
            }
        },
		ObjectProperty (path){
			// 收集通过 key:function(){ } 方式写的Vue methods函数
            if (isGeneralMethods(path)) {
				const name = path.node.key.name;
				collectGeneralMethods(path, collect, state, name);
            }
		}
    });

	logi('CollectState[3]      ');

    let renderJSXArgs = null;
	let styleArgs = null;
	let styleLink = null;
    if (isSFC) {
		renderJSXArgs = traverseTemplate(component.template, state);
		if(component.styles && component.styles.length > 0){
			let styleStr = '';
			let link = new Array();	
			component.styles.forEach((style) => {
				let content = style.content ? style.content.trim() : '';
				if(content){
					styleStr += cssFaultTolerant(content);
				}else if(!content && style.src){
					let src = resolve(dirname(filename), style.src);
					saveAsJson(src); // stml中通过style标签src引入的css，生成同名的json文件，APP中通过增加.json后缀引用
					// 处理 src引入

					//由于最后组件会统一编译到一个文件中,将组件中的css地址进行转换
					var relativePath = relative(dirname(global.inputfile), dirname(filename));
					link.unshift(join(relativePath, style.src).replace(/\\/g,'/'));
				}
				if(style.scoped){
					// 处理 scoped
				}
			})
			if(styleStr){
				styleArgs = JSON.stringify(css2Json(styleStr));
			}
			if(link.length > 0){
				styleLink = JSON.stringify(link);
			}
		}
    }
	logi('CollectState[4]      ');

	/* ----------------------------- 开始将暂存的state, collect导出到APICloud标准vdom ------------------------------ */

	let componentName = state.name;
	if(!componentName){
		componentName = basename(filename);
	}
	const className = normalizedComponentName(componentName);
    // AST for define component，定义apivm组件模板，用于将state、collect、renderJSXArgs收集的代码进行组装
	// const tpl = `define('${componentName}', class extends Component {});`;
	if (isImport && global.platform == 'web' ) {
	    var tpl = `function ${className}_create(){ 
	    	var api = Object.create(window.api);
			class ${className} extends Component {};
				${styleArgs ? `${className}.css=${styleArgs};` : ''}
				${styleLink ? `${className}.csslink=${styleLink};` : ''}
				${isSFC ? `apivm.define('${componentName}', ${className});` : ''} 
			return 	${className}
		}`;
	} else {
	    var tpl = `
	    	class ${className} extends Component {};
			${styleArgs ? `${className}.css=${styleArgs};` : ''}
			${styleLink ? `${className}.csslink=${styleLink};` : ''}
			${isSFC ? `apivm.define('${componentName}', ${className});` : ''} `;
	}


    const rast = babylon.parse(tpl, {
        sourceType: 'module'
    });

	logi('Rebuilt[1]           ');

    babelTraverse(rast, {

        Program (path) {
            genImports(path, collect, state);
			genDeclarations(path, collect);
        },
    
        ClassBody (path) {
            genConstructor(path, state, filename);
            genStaticProps(path, state);
            genClassMethods(path, collect);
            isSFC && genSFCRenderMethod(path, state, renderJSXArgs);
        }
    });

	logi('Rebuilt[2]           ');

    if (isSFC) {
        // replace custom element/component
        babelTraverse(rast, {

            ClassMethod (path) {
                if (path.node.key.name === 'render') {
                    path.traverse({
                        JSXIdentifier (path) {
                            if (t.isJSXClosingElement(path.parent) || t.isJSXOpeningElement(path.parent)) {
                                const node = path.node;
                                const componentName = state.components[node.name] || state.components[normalizedComponentName(node.name)];
                                if (componentName) {
                                    path.replaceWith(t.jSXIdentifier(componentName));
                                    path.stop();
                                }
                            }
                        }
                    });
                }
            }
        });
    }

	logi('Rebuilt[3]           ');

    let { code, map } = generate(rast, {
        retainLines: true,
		//sourceMaps: true
    });

	logi('Rebuilt[4]           ');

	// 引入的组件，不做render，顶级page则render到body
	if(!isImport){
		code += `apivm.render(<${componentName} />, 'body');`;
	}else{
		// 组件增加export，支持"import Xxx from './xxx.stml';"方式引用
		if(global.platform == 'web' ){
			code += `const ${className} = ${className}_create(); export default ${className};`;
		}else{
			code += `export default ${className}`;
		}
	}
	
	code = bublejsx(code);

	// code = format(code); // 注销，在build-config中格式化

	logi('Transpile: exit      ');

	return code;
}

/*
	编译stml代码为avm代码
	
	@ code stml代码字符串
	@ filename 代码对应的文件地址
	@ isSFC 是否是stml文件
	@ isImport 是否是来自import
*/
function transformSource(code, filename, isSFC, isImport){

	return transformForComponent(code, filename, isSFC, isImport);
};

module.exports = transformSource;
