/* Copyright 2020 APICloud Inc. All Rights Reserved.


 */

const babelTraverse = require('@babel/traverse').default;
const t = require('@babel/types');

const {
	warn,
	error, 
	logast
} = require('./utils');
const collectVueProps = require('./vue-props');
const collectVueComputed = require('./vue-computed');

/**  
 * Collect vue component state(data prop, props prop & computed prop)
 * Don't support watch prop of vue component
 */
exports.collectProps = function collectProps (ast, state) {
    babelTraverse(ast, {

        Program (path) {
            const nodeLists = path.node.body;
            let count = 0;
    
            for (let i = 0; i < nodeLists.length; i++) {
                const node = nodeLists[i];
                // const childPath = path.get(`body.${i}`);
                if (t.isExportDefaultDeclaration(node)) {
                    count++;
                }
            }
    
            if (count > 1 || !count) {
                const msg = !count ? ' Must hava one' : ' Only one';
                warn(`${msg} export default declaration in your stml component file`);
                //process.exit();
            }
        },

        ObjectProperty (path) {
            const parent = path.parentPath.parent;
            const name = path.node.key.name;
            if (parent && t.isExportDefaultDeclaration(parent)) {
                if (name === 'name') {
                    if (t.isStringLiteral(path.node.value)) {
						const value = path.node.value.value;
						if(value && value.trim()){ // 在组件name不能为空。add by moral.li 2019/12/06
							state.name = value;
						}else{
							error(` The component value of name prop should not be empty.`);
						}
                        // state.name = path.node.value.value;
                    } else {
                        error(`The value of name prop should be a string literal.`);
                    }
                } else if (name === 'props') {
                    collectVueProps(path, state);
                    path.stop();
                }
            }
        }
    });
};

/**  
 * Collect vue component data Method
 */
exports.collectData = function collectData (ast, state) {
    babelTraverse(ast, {
        ObjectMethod (path) {
            const parent = path.parentPath.parent;
            const name = path.node.key.name;

            if (parent && t.isExportDefaultDeclaration(parent)) {
                if (name === 'data') {
                    const body = path.node.body.body;
                    state.data['_statements'] = [].concat(body);

                    let propNodes = {};
                    body.forEach(node => {
                        if (t.isReturnStatement(node)) {
                            propNodes = node.argument.properties;
                        }
                    });

                    propNodes.forEach(propNode => {
                        state.data[propNode.key.name] = propNode.value;
                    });
                    path.stop();
                }
            }
        }
    });
};

/**  
 * Collect vue component computed prop
 */
exports.collectComputed = function collectComputed (ast, state) {
    babelTraverse(ast, {
        ObjectProperty (path) {
            const parent = path.parentPath.parent;
            const name = path.node.key.name;
            if (parent && t.isExportDefaultDeclaration(parent)) {
                if (name === 'computed') {
                    collectVueComputed(path, state);
                    path.stop();
                }
            }
        }
    });
};

/**  
 * Collect vue component components prop
 */
exports.collectComponents = function collectComponents (ast, state) {
    babelTraverse(ast, {
        ObjectProperty (path) {
            const parent = path.parentPath.parent;
            const name = path.node.key.name;
            if (parent && t.isExportDefaultDeclaration(parent)) {
                if (name === 'components') {
                    // collectVueComputed(path, state);
                    const props = path.node.value.properties;
                    props.forEach(prop => {
                        state.components[prop.key.name] = prop.value.name;
                    });
                    path.stop();
                }
            }
        }
    });
};