/* Copyright 2020 APICloud Inc. All Rights Reserved.
 

 */

/*
*
* 编译stml文件for webapp入口。用于编译某目录下所有.stml文件为spa应用。
*/

'use strict';

const fs = require('fs');

const minify = require('./htmlminifier.min');

const {
	statSync,
	writeFileSync,
	existsSync,
	readFileSync
} = require('fs');

const path = require('path');
const {
	join,
	basename,
	dirname,
	extname,
	relative,
	resolve
} = require('path');
const build = require('./build');

const { 
	log,
	warn,
	error,
	magenta,
	notifyStart, 
	notifyEnd,
	state
} = require('./tools/client');

const {
	findAll,
	findName,
	findContent
} = require('./tools/config-parse');

const {
	pathExists, 
	copyDirSync,
	copyFileSync,
	mkpath,
	scanStmlSync
} = require('./tools');

const {
	makeTabbarInfo
} = require('./tools/tabbar-parse');

const DEFAULT_NAMESPACE = 'Tools.Toweb.';
function info(msg){
	log(msg, DEFAULT_NAMESPACE);
}

/**/
const process = require('process');
process.on('unhandledRejection', (reason, p) => {
	console.log('Compile Unhandled Rejection at: \n', reason);
});

async function compileFolder(rootFolder, outFolder) {
	performStart();
	info('WebApp Compile: enter. ');
	const stat = state();
	if(!existsSync(rootFolder)){
		stat.msg = 'ERROR: ' + rootFolder + ' not exist!';
		warn(stat.msg);
		performEnd(stat);
		return stat;
	}
	if(!outFolder){
		outFolder = rootFolder + '/out-web';
	}
	if(!pathExists(outFolder)){
		mkpath(outFolder);
	}
	const lists = scanStmlSync(rootFolder);
	if(!lists || 0 == lists.length){
		stat.msg = 'WARN: ' + rootFolder + ' is empty';
		warn(stat.msg);
		performEnd(stat);
		return stat;
	}

	copyIndexTemplate(rootFolder, outFolder);
	copyStatic(rootFolder, outFolder);

	stat.total = lists.length;
	const doBuild = async function(filename){
		let sub = filename.replace(rootFolder, '').replace('.stml', '');
		let outfile = path.join(outFolder, sub + '.js');
		try{
			await build(filename, outfile, 'web');
			stat.ok.push(filename);
		}catch(e){
			stat.err.push(filename);
			error('Compile Folder process: ' + (e.stack ? e.stack : e) + '\n at ' + path.basename(filename),undefined,{codeFrame:{filename,e}});
		}
		await doForEach();
	}
	const doForEach = async function(){
		if(lists && lists.length > 0){
			await doBuild(lists.shift());
		}
	}
	await doForEach();
	info('WebApp Compile: exit. ');
	performEnd(stat);
	return stat;
}

//拷贝静态资源 css 和 image
function copyStatic(rootFolder, outFolder){
	const dirs = ['image', 'images', 'static', 'icon', 'css'];
	dirs.forEach((dir) => {
		const src = join(rootFolder, dir);
		if(pathExists(src)){
			const desc = join(outFolder, dir);
			copyDirSync(src, desc);
		}
	});
}

// 拷贝web框架模板，index入口页
function copyIndexTemplate(rootFolder, outFolder){
	const tmplPath = join(__dirname, './template/web');
	copyDirSync(tmplPath, outFolder);
	var indexHtml = readFileSync(join(outFolder, './pages/index/index.html')).toString();
	indexHtml = minifyHtml(indexHtml)


	const configDir = join(rootFolder, './config.xml');
	const data = readFileSync(configDir).toString();
	const info = findAll(data);


	const ename = extname(info.src);
	const appName = info.name
	if(ename === '.stml'){
		var relativePath = relative('pages/index',info.src).replace(/\\/g,'/');
		var jsonData={
			appName: appName,
			rootPage: relativePath
		}
	}else if(ename === '.json'){
		var path = join(dirname(configDir), info.src);
		var jsonData = readFileSync(path).toString();
		jsonData = JSON.parse(jsonData);
		jsonData.appName = appName;
		jsonData.tabBar.frames.forEach(item=>{
			item.url = relative('pages/index', item.url).replace(/\\/g,'/');
		})
		jsonData.tabBar.list.forEach(item=>{
			item.iconPath = relative('pages/index', item.iconPath).replace(/\\/g,'/');
			item.selectedIconPath = relative('pages/index', item.selectedIconPath).replace(/\\/g,'/');
		})
	}

	indexHtml = '<html>\r\n<script> var _avm_root_config='+JSON.stringify(jsonData)+'</script>'+indexHtml.substr(6);
	writeFileSync(join(outFolder, './pages/index/index.html'), indexHtml);
}


function minifyHtml(source) {
	source = minify(source,{
		removeIgnored: true,
        removeComments: true,
        collapseWhitespace: true,
        collapseBooleanAttributes: true,
        minifyJS: true,
        minifyCSS: true
	})
    
	return source
}


function performStart(_msg){
	notifyStart(_msg);
}

function performEnd(stat){
	notifyEnd(stat);
}

exports.compileFolder = compileFolder