/* Copyright 2020 APICloud Inc. All Rights Reserved.
 

 */

/*
*
* 编译stml文件for 小程序入口。用于编译某stml文件或者某目录下所有.stml文件为小程序代码。
*/

'use strict';

const fs = require('fs');
const {
	statSync,
	writeFileSync,
	readFileSync
} = require('fs');

const path = require('path');
const {
	join,
	basename,
	dirname,
	resolve,
	isAbsolute,
	extname
} = require('path');
const build = require('./build');

const { 
	log,
	warn,
	error,
	magenta,
	notifyStart, 
	notifyEnd,
	state
} = require('./tools/client');

const {
	pathExists, 
	copyDirSync,
	copyFileSync,
	mkpath,
	scanStmlSync,
	subStmlPath
} = require('./tools');

const {
	makePageIndex,
	makePageJs,
	makePageJson,
	makePageCss,
	makePageXml
} = require('./tools/mp');

const {
	findAll,
	findName,
	findContent
} = require('./tools/config-parse');

const {
	makeTabbarInfo
} = require('./tools/tabbar-parse');

/**/
const process = require('process');
process.on('unhandledRejection', (reason, p) => {
	console.log('Compile Unhandled Rejection at: \n', reason);
	error('Compile Unhandled Rejection at: \n' + reason);
});
/**/
if (typeof String.prototype.endsWith !== 'function') {
    String.prototype.endsWith = function(suffix) {
        return this.indexOf(suffix, this.length - suffix.length) !== -1;
    };
}

const DEFAULT_NAMESPACE = 'Tools.Tominiapp.';

function info(msg){
	log(msg, DEFAULT_NAMESPACE);
}

async function compileStml(inputfile, outfile) {
	
	return await build(inputfile, outfile, 'miniapp');
}

var appJson;
var appJsStr;
var projectJson;

// 拷贝小程序工程模板
function copyMpTemplate(out){
	const tmplPath = join(__dirname, './template/mp');
	copyDirSync(tmplPath, out);
	appJson = JSON.parse(readFileSync(join(out, './app.json')).toString());
	appJsStr = readFileSync(join(out, './app.js')).toString();
	projectJson = JSON.parse(readFileSync(join(out, './project.config.json')).toString());
}

// 拷贝项目中所有静态资源至小程序工程
function copyAsset(rootFolder, outRootFolder){
	const dirs = ['image', 'images', 'static', 'icon'];
	dirs.forEach((dir) => {
		const src = join(rootFolder, dir);
		if(pathExists(src)){
			const desc = join(outRootFolder, dir);
			copyDirSync(src, desc);
		}
	});
}

// 保存编译结果至wxml|wxss|js|json
function savePage(result, outDir, filepath, outRootFolder){

	mkpath(outDir);
	const fname = basename(filepath, '.stml');
	
	const jsCode = result.code;
	const pageCode = makePageJs(jsCode);
	const pagePath = join(outDir, './page.js');
	writeFileSync(pagePath, pageCode);

	const indexCode = makePageIndex();
	const indexPath = join(outDir, `./${fname}.js`);
	writeFileSync(indexPath, indexCode);

	const jsonCode = makePageJson(filepath);
	const jsonPath = join(outDir, `./${fname}.json`);
	writeFileSync(jsonPath, JSON.stringify(jsonCode, null, 4));

	const cssCode = makePageCss(result, filepath, outRootFolder);
	const cssPath = join(outDir, `./${fname}.wxss`);
	cssCode && writeFileSync(cssPath, cssCode);

	const xmlCode = makePageXml();
	const xmlPath = join(outDir, `./${fname}.wxml`);
	writeFileSync(xmlPath, xmlCode);
	
	updatePageInfo(fname);

	info('Save Page: ' + basename(filepath));
}

// 更新page信息至app.json
function updatePageInfo(basename){
	if(appJson){
		const page = `pages/${basename}/${basename}`;
		appJson.pages.push(page);
	}
}

// 更新app.json和project.config.json信息
function updateAppInfo(inFolder, outFolder, options){
	const configDir = join(inFolder, './config.xml');
	const data = readFileSync(configDir).toString();
	const info = findAll(data);
	if(!info){
		return;
	}
	if(info.name){
		appJson.window.navigationBarTitleText = info.name;
		projectJson.projectname = info.name;
	}
	if(info.src){
		var mainpage = info.src;
		const ename = extname(mainpage);
		var firstPage = null;
		if(ename === '.stml'||ename === ''){
			firstPage = mainpage.replace('.stml', '');
		}else if(ename === '.json'){
			mainpage = join(dirname(configDir), mainpage);
			const jsonData = readFileSync(mainpage).toString();
			const tabbar = makeTabbarInfo(jsonData);
			if(tabbar){
				appJson.tabBar = tabbar;
				appJsStr = 'var avmConfig = { tabBarList: ' + JSON.stringify(tabbar.list) + '};\r\n' + appJsStr
				firstPage = tabbar.list[0].pagePath;
			}
		}
		if(firstPage){
			var index = appJson.pages.indexOf(firstPage);
			if(index > 0){
				appJson.pages.splice(index, 1);
				appJson.pages.unshift(firstPage);
			}
			appJson.entryPagePath = firstPage;
		}
	}
	if(options){
		if(options.appid){
			projectJson.appid = options.appid;
		}
		if(options.projectname){
			projectJson.projectname = options.projectname;
		}
		if(options.description){
			projectJson.description = options.description;
		}
	}
}

function compileBegin(inFolder, outFolder, options){
	
}

function compileEnd(inFolder, outFolder, options){
	updateAppInfo(inFolder, outFolder, options);
	const appjsonPath = join(outFolder, './app.json');
	writeFileSync(appjsonPath, JSON.stringify(appJson, null, '\t'));
	const projectjsonPath = join(outFolder, './project.config.json');
	writeFileSync(projectjsonPath, JSON.stringify(projectJson, null, 4));
	const appjsPath = join(outFolder, './app.js');
	writeFileSync(appjsPath, appJsStr);
	copyAsset(inFolder, outFolder);
}

async function compileOne(filename, outRootFolder, options) {
	const rootFolder = subStmlPath(filename);
	const sub = dirname(filename.replace(rootFolder, ''));
	const outDir = join(outRootFolder, sub);
	const result = await compileStml(filename);
	savePage(result, outDir, filename, outRootFolder);
}

/*
	编译单个stml文件为小程序代码
	@ filename stml文件绝对路径
	@ outRootFolder 编译后的小程序工程代码输出目标目录根目录
*/
async function compileSingle(filename, outRootFolder, options) {
	performStart();
	const stat = state();
	try{
		await compileOne(filename, outRootFolder, options);
		stat.total = 1;
		stat.ok.push(filename);
	}catch(e){
		stat.err.push(filename);
		error('Compile stml: at ' + (e.stack ? e.stack : e) + '\n at ' + basename(filename));
	}
	performEnd(stat);
}

/*
	编译stml文件列表为小程序代码
	@ lists stml文件列表，其中每个stml文件路径必须为绝对路径
	@ outRootFolder 编译后的小程序工程代码输出目标目录根目录
*/
async function compileList(lists, outRootFolder, options) {
	performStart();
	const stat = state();
	if(!lists || 0 == lists.length){
		stat.msg = 'stml list is empty!';
		warn(stat.msg);
		performEnd(stat);
		return stat;
	}
	stat.total = lists.length;
	const doBuild = async function(item){
		const filename = item;
		try{
			await compileOne(filename, outRootFolder, options);
			stat.ok.push(filename);
		}catch(e){
			stat.err.push(filename);
			error('Compile List process: ' + (e.stack ? e.stack : e) + '\n at ' + path.basename(filename));
		}
		await doForEach();
	}
	const doForEach = async function(){
		if(lists && lists.length > 0){
			await doBuild(lists.shift());
		}
	}
	await doForEach();
	performEnd(stat);
	return stat;
}

/*
	同步编译widget目录为小程序项目工程
	@ rootFolder widget根目录
	@ outRootFolder 编译后的小程序工程代码输出目标目录根目录
*/
async function compileFolder(rootFolder, outRootFolder, options) {
	performStart();
	info('MiniApp Compile: enter. ');
	if(!outRootFolder){
		outRootFolder = rootFolder + '/out-mp';
	}
	compileBegin(rootFolder, outRootFolder, options);
	if(!pathExists(outRootFolder)){
		mkpath(outRootFolder);
	}
	copyMpTemplate(outRootFolder);
	const stat = state();
	var lists = scanStmlSync(rootFolder);
	stat.total = lists.length;
	const doCompile = async function(filename) {
		const sub = dirname(filename.replace(rootFolder, ''));
		const outDir = join(outRootFolder, sub);
		try{
			const result = await compileStml(filename);
			savePage(result, outDir, filename, outRootFolder);
			stat.ok.push(filename);
		}catch(e){
			stat.err.push(filename);
			error('Compile stml: at ' + (e.stack ? e.stack : e) + '\n at ' + basename(filename),undefined,{codeFrame:{filename,e}});
		}
		await doForEach();
	}

	const doForEach = async function(){
		if(lists && lists.length > 0){
			await doCompile(lists.shift());
		}
	}
	await doForEach();
	compileEnd(rootFolder, outRootFolder, options);
	info('MiniApp Compile: exit. ');
	performEnd(stat);	
	return true;
}

function performStart(_msg){
	notifyStart(_msg);
}

function performEnd(stat){
	notifyEnd(stat);
}

// module.exports = compileFolder;

exports.compileSingle = compileSingle;
exports.compileList = compileList;
exports.compileFolder = compileFolder;