/* Copyright 2020 APICloud Inc. All Rights Reserved.
 

 */

/*
*
* 编译stml文件for native app 入口。用于编译某stml文件或者某目录下所有.stml文件为app js代码。
*/

'use strict';

const rollupPluginutils = require('rollup-pluginutils');
const fs = require('fs');
const path = require('path');
const statSync = fs.statSync;
const existsSync = fs.existsSync;
const build = require('./build');
const { 
	log, 
	error, 
	warn,
	notifyStart, 
	notifyEnd,
	state
} = require('./tools/client');

const {
	scanStmlSync
} = require('./tools');

/**/
const process = require('process');
process.on('unhandledRejection', (reason, p) => {
	console.log('Compile Unhandled Rejection at: \n', reason);
});

async function _build(inputfile, outfile) {
	
	return await build(inputfile, outfile, 'app');
}

async function _buildlist(lists) {
	const stat = state();
	if(!lists || 0 == lists.length){
		warn('stml list is empty!');
		stat.msg = 'stml list is empty!';
		return stat;
	}
	stat.total = lists.length;
	const doBuild = async function(item){
		const filename = item.input;
		const outfile = item.output;
		try{
			await _build(filename, outfile);
			stat.ok.push(filename);
		}catch(e){
			stat.err.push(filename);
			error('Compile List process: ' + (e.stack ? e.stack : e) + '\n at ' + path.basename(filename));
		}
		await doForEach();
	}
	const doForEach = async function(){
		if(lists && lists.length > 0){
			await doBuild(lists.shift());
		}
	}
	await doForEach();
	return stat;
}

/*
 * 编译单个stml文件
 * @ inputfile stml文件地址
 * @ outfile 编译后的js文件存储地址。可选
 */
async function buildSingleSync(inputfile, outfile) {
	performStart();
	const stat = state(); 
	await _build(inputfile, outfile);
	stat.total = 1;
	stat.ok.push(inputfile);
	performEnd(stat);
	return stat;
}

/* 
 * 编译stml目录，通常用于全量同步和云编译
 * @ rootFolder 带stml文件的根目录，例如pages目录
 * @ outRootFolder 编译后的js文件存放目标目录，例如.bin目录
 */
async function buildFolderSync(rootFolder, outRootFolder) {
	performStart();
	const stat = state();
	if(!existsSync(rootFolder)){
		stat.msg = 'ERROR: ' + rootFolder + ' not exist!';
		warn(stat.msg);
		performEnd(stat);
		return stat;
	}
	const lists = scanStmlSync(rootFolder);
	if(!lists || 0 == lists.length){
		stat.msg = 'WARN: ' + rootFolder + ' is empty';
		warn(stat.msg);
		performEnd(stat);
		return stat;
	}
	stat.total = lists.length;
	const doBuild = async function(filename){
		let sub = filename.replace(rootFolder, '').replace('.stml', '');
		let outfile = path.join(outRootFolder, sub + '.js');
		try{
			await _build(filename, outfile);
			stat.ok.push(filename);
		}catch(e){
			stat.err.push(filename);
			error('Compile Folder process: ' + (e.stack ? e.stack : e) + '\n at ' + path.basename(filename),undefined,{codeFrame:{filename,e}});
		}
		await doForEach();
	}
	const doForEach = async function(){
		if(lists && lists.length > 0){
			await doBuild(lists.shift());
		}
	}
	await doForEach();
	performEnd(stat);
	return stat;
}

/* 
 * 编译stml列表，通常用于增量同步
 * @ lists
 *    -item: {input:'', output:''}
 */
async function buildListSync(lists) {
	performStart();
	let stat = state();
	if(!lists || 0 == lists.length){
		stat.msg = 'WARN: stml list is empty!';
		warn(stat.msg);
		performEnd(stat);
		return stat;
	}
	stat = await _buildlist(lists);
	performEnd(stat);
	return stat;
}

function performStart(_msg){
	notifyStart(_msg);
}

function performEnd(stat){
	notifyEnd(stat);
}

exports.buildSingle = buildSingleSync;
exports.buildList = buildListSync;
exports.buildFolder = buildFolderSync;