/* Copyright 2020 APICloud Inc. All Rights Reserved.
 

 */

/*
*
* 用于编译单个.stml文件
* http://rollupjs.org/guide/en/#rolluprollup
*/

'use strict';

const rollup = require('rollup');
const {
	basename,
	join,
	resolve,
	dirname
}= require('path');

const formatOptions = require('./build-config');
const { 
	log,
	local,
	magenta
} = require('./tools/client');

if (typeof String.prototype.startsWith != 'function') {
	String.prototype.startsWith = function (prefix){
		return this.slice(0, prefix.length) === prefix;
	};
}

// for小程序的js代码和css资源需分离和处理
function severmp(outputBundle, inputfile){
	if(1 === outputBundle.length){
		const chunkOrAsset = outputBundle[0];
		const code = chunkOrAsset.code;
		const map = chunkOrAsset.map;
		log('Compile exit with return code. ');
		return { code, map };
	}else{
		var code = ''; // 编译完的js代码
		var css = {    // 分离的css代码
			code: '',
			link: []
		};
		for (var i= outputBundle.length - 1; i >= 0; i--) {
			const chunkOrAsset = outputBundle[i];
			if (chunkOrAsset.type === 'asset') {
				/*
				  lib/sfc-to-dsl/index.js 中将stml中的css分离出来，
				  以cssimport-xxx和cssdata-xxx存储在chunkOrAsset
				*/
				if(chunkOrAsset.fileName.startsWith('cssdata-')){
					css.code = css.code + '\n' + chunkOrAsset.source;
				}
				if(chunkOrAsset.fileName.startsWith('cssimport-')){
					css.link.push(chunkOrAsset.source);
				}
			} else {
				code += chunkOrAsset.code;
			}
		}
		log('Compile exit with return chunk code. ');
		return { code, css };
	}
}

/*
	@ inputfile stml文件地址
	@ outfile 编译后的js文件存储地址。可选
	@ 编译类型，app | webapp | miniapp
*/
async function build(inputfile, outfile, platform) {
	global.inputfile = inputfile;
	const miniapp = 'miniapp' === platform;
	const options = formatOptions({
		input: inputfile,
		output: outfile,
		platform: platform
	});
	
	log('Compile: enter. ' + (local ? magenta(basename(options.input)) : basename(options.input)));
	const bundle = await rollup.rollup(options);
	
	if(options.output.file){
		// 编译app，结果直接输出到output
		await bundle.write(options.output);
		log('Compile exit.       ');
		return;
	}
	if(miniapp){
		// 编译小程序，css资源需分离和处理
		const { output } = await bundle.generate(options);
		return severmp(output, inputfile);
	}
	log('Compile exit with no output. ');
}

module.exports = build;