/* Copyright 2020 APICloud Inc. All Rights Reserved.


 */

/*
* 编译配置，rollup扩展插件在此接入
*
* 编译jsx语法目前用的是babel-plugin-transform-reac-jsx 换成 babel-plugin-transform-vue-jsx 更能兼容Vue？
*
*/

'use strict';
// 后期Babel换成Buble？Buble支持得特性更少，比如不支持static关键字
const babel = require('rollup-plugin-babel');
const nodeResolve = require('rollup-plugin-node-resolve');	// 识别代码中引入的来自node_modules中的包
const commonjs = require("rollup-plugin-commonjs");			// CJS -> ESM。node_modules中的包大部分都是commonjs格式的，必须先转为ES6语法
const json = require("rollup-plugin-json");
const { uglify } = require("rollup-plugin-uglify");
const prettier = require('rollup-plugin-prettier');
const buble = require("@rollup/plugin-buble");

const sfcPlugin = require('./sfc-to-dsl');
const cssPlugin = require('./style-to-json');

// sourcemap | treeshake将来做成配置，允许开发者定制

function initOptions(opt){
	return {
		treeshake: opt.treeshake, // 是否优化无用代码
		input: opt.input,
		output: {
			format: 'iife',
			file: opt.output,
			//name: objectName
			sourcemap: opt.sourcemap,
			strict: false
		},
		plugins: [
			json(),
			cssPlugin({
				split: true,
				include: [/\.css$/i],
				platform: opt.platform
			}),
			nodeResolve({
				mainFields: ['main'], //mainFields: ['module', 'main']
				//extensions: [ '.stml', '.js', '.jsx', '.json' ]
			}),
			commonjs({
				include: 'node_modules/**'
			})
		]
	}
};

/*
	Babel只用于js形式写的代码进行编译（能支持更多的写法，比如static等），stml编译使用Buble
	Babel options: https://babeljs.io/docs/en/options
*/
function babelPlugin(){

	return babel({
			sourcemap: true,
			exclude: 'node_modules/**',
			//include: [/\.stml$/i],
			extensions:[".js", ".jsx", ".es", ".mjs"], // 只编译js
			babelrc: false,
			retainLines: true,
			presets: [
				['@babel/preset-env', {
					modules: false,
					loose: true,
					targets: {
						browsers: ['last 2 versions', 'Chrome >= 60']
					}
				}],
			],
			plugins: [
				'@babel/plugin-proposal-object-rest-spread',					// 支持编译ES6三目运算符
				['@babel/plugin-proposal-decorators', { 'legacy': true }],		// 支持编译ES7装饰器
				'@babel/plugin-proposal-class-properties',						// 支持编译ES6 static关键字
				["@babel/plugin-transform-react-jsx", { "pragma":"apivm.h" }]	// jsx语法编译为vdom的h函数。注意：不能用_h等_开头的函数，列表map编译时会被覆盖
			]
		});
}

/*
	Buble：https://buble.surge.sh/guide/
*/
function bublePlugin(){
	return buble({
		include: [/\.stml$/i],
		target: { chrome: 48, firefox: 44 },
		transforms: {
			arrow: true,
			classes: true,
			computedProperty: true,
			conciseMethodProperty: true,
			defaultParameter: true,
			destructuring: true,
			dangerousForOf: true,
			exponentiation: true,
			generator: true,
			letConst: true,
			modules: false,
			moduleImport: false, // Buble目前暂不支持import，
			numericLiteral: true,
			objectRestSpread: true,
			parameterDestructuring: true,
			spreadRest: true,
			stickyRegExp: true,
			templateString: true,
			unicodeRegExp: true
		},
		jsx:'apivm.h',
		jsxFragment: "apivm.f",
		objectAssign: true, // 'Object.assign'
		namedFunctionExpressions: false  // class.prototype.xxx = function xxx(){} -> class.prototype.xxx = function(){}
	});
}

/*
	格式化代码
*/
function prettierPlugin(){
	var options = {
		printWidth: 80,					// 超过最大值换行
		tabWidth: 1,					// 缩进字节数
		useTabs: true,					// 是否缩进使用tab，不使用空格
		semi: true,						// 是否句尾添加分号
		singleQuote: false,				// 是否使用单引号代替双引号
		trailingComma: "none",			// 是否在对象或数组最后一个元素后面加逗号（在ES5中加尾逗号）
		bracketSpacing: false,			// 是否在对象，数组括号与文字之间加空格 "{ foo: bar }"
		jsxBracketSameLine: false,		// 是否在jsx中把'>'单独放一行
		jsxSingleQuote: false,			// 是否在jsx中使用单引号代替双引号
		stylelintIntegration: false,	// 是否使用stylelint的代码格式进行校验
		eslintIntegration: false,		// 是否使用eslint的代码格式进行校验
		parser: "babel"
	};
	return prettier(options);
}

/*
	压缩混淆代码
	UglifyJS options: https://github.com/mishoo/UglifyJS/blob/master/README.md#minify-options
*/
function uglifyPlugin(){
	var options = {
		compress: {
			conditionals: false,
			sequences: false,
			join_vars: false,
			collapse_vars: true,
			evaluate: true,
			unsafe: true,
			loops: false,
			keep_fargs: false,
			pure_getters: true,
			unused: true,
			dead_code: true
		},
		mangle: {
			// mangle options
			properties: {
				// mangle property options
				regex: /^(normalizedNodeName)/
			}
		},
		sourcemap: false,
		nameCache: null, // or specify a name cache object
		toplevel: false,
		ie8: false,
		warnings: false,
	};
	return uglify(options);
}

const formatOptions = function (opt){
	opt.sourcemap = false;
	//opt.treeshake = false;
	const options = initOptions(opt);
	const sfcHandle = sfcPlugin({
		needMap: opt.sourcemap,
		platform: opt.platform,
		include: [/\.stml$/i] // 只编译stml
	});
	const babelHandle = babelPlugin();
	const bubleHandle = bublePlugin();
	const uglifyHandle = uglifyPlugin();
	const prettierHandle = prettierPlugin();
	options.plugins.unshift(sfcHandle);
	//options.plugins.push(bubleHandle); // 见buble-jsx.js，编译class属性/静态属性不成功
	options.plugins.push(babelHandle);		// 耗时
	options.plugins.push(prettierHandle);	// 耗时
	//options.plugins.push(uglifyHandle);	// 耗时

	if('miniapp' === opt.platform){
		options.onwarn = function(warning, warn) {
			if (warning.code === 'INPUT_HOOK_IN_OUTPUT_PLUGIN') {
				return;
			}
			warn(warning);
		}
	}

	return options;
}

module.exports = formatOptions;
