const vscode = require('vscode');
const { compileAll } = require('./command/complieAll');
const { compile } = require('./command/compileSingle');
const { compileMiniapp } = require('./command/compileMiniapp');
const fs = require('fs');
const path = require('path');
const { parseStml } = require('./parseStml');
const { getPath } = require('./common/projectUtil');
const os = require('os');

/**
 * @param {vscode.ExtensionContext} context
 */
function activate(context) {

	compileAll(context);
	compile(context);
	compileMiniapp(context);

	cacheStmlTag();

	registerListener(context);
}

async function cacheStmlTag() {
	const workspaceFolders = vscode.workspace.workspaceFolders;
	if (workspaceFolders) {
		for (let index = 0; index < workspaceFolders.length; index++) {
			const folder = workspaceFolders[index];
			if (fs.existsSync(path.join(folder.uri.fsPath, 'config.xml')) && getPath(folder.uri.fsPath)) {
				writeStmlTag(folder.uri.fsPath);
			}
		}
	}
}

/**
 * 写入stml标签
 * @param {string} folder
 */
async function writeStmlTag(folder) {
	const pattern = new vscode.RelativePattern(folder, '**/*.stml');
	const uris = await vscode.workspace.findFiles(pattern);
	if (uris.length > 0) {
		const stat = {};
		for (let index = 0; index < uris.length; index++) {
			const uri = uris[index];
			parseStml(uri, stat);
		}
		const infoRoot = path.join(folder, '.yonstudio');
		if (!fs.existsSync(infoRoot)) {
			fs.mkdirSync(infoRoot);
		}
		fs.writeFileSync(path.join(infoRoot, '.labelinfo'), JSON.stringify(stat));
	}
}

/**
 * @param {vscode.ExtensionContext} context
 */
function registerListener(context) {
	// 监听文件保存
	context.subscriptions.push(vscode.workspace.onDidSaveTextDocument((e) => {
		const uri = e.uri;
		if (uri.fsPath.endsWith('.stml')) {
			const projectPath = getPath(uri.fsPath);
			if (projectPath) {
				const stat = {};
				parseStml(uri, stat);
				const labelJson = fs.readFileSync(path.join(projectPath, '.yonstudio/.labelinfo'), { encoding: 'utf-8' });
				const label = JSON.parse(labelJson);
				for (const skey in stat) {
					const lc = label[skey];
					const sc = stat[skey];
					if (!lc) {
						for (const key in label) {
							const c2 = label[key];
							if (c2 === sc) {
								delete label[key];
								label[skey] = sc;
							}
						}
					}
				}
				fs.writeFileSync(path.join(projectPath, '.yonstudio/.labelinfo'), JSON.stringify(label));
			}
		}
	}));

	// 监听文件删除
	context.subscriptions.push(vscode.workspace.onDidDeleteFiles((e) => {
		const uris = e.files;
		for (let index = 0; index < uris.length; index++) {
			const uri = uris[index];
			if (uri.fsPath.endsWith('.stml')) {
				const projectPath = getPath(uri.fsPath);
				if (projectPath) {
					const labelJson = fs.readFileSync(path.join(projectPath, '.yonstudio/.labelinfo'), { encoding: 'utf-8' });
					const label = JSON.parse(labelJson);
					for (const key in label) {
						if (label[key] === uri.fsPath) {
							delete label[key];
						}
					}
					fs.writeFileSync(path.join(projectPath, '.yonstudio/.labelinfo'), JSON.stringify(label));
				}
			}
		}
	}));

	// 监听添加folder
	context.subscriptions.push(vscode.workspace.onDidChangeWorkspaceFolders((e) => {
		const addFolders = e.added;
		for (let index = 0; index < addFolders.length; index++) {
			const folder = addFolders[index];
			if (fs.existsSync(path.join(folder.uri.fsPath, 'config.xml')) && getPath(folder.uri.fsPath)) {
				writeStmlTag(folder.uri.fsPath);
			}
		}
	}));
}

// @ts-ignore
exports.activate = activate;

function deactivate() { }

module.exports = {
	// @ts-ignore
	activate,
	deactivate
};
