/* Copyright 2020 APICloud Inc. All Rights Reserved.
 */

const fs = require('fs');
const path = require('path');
const os = require('os');
const vscode = require('vscode');
const xml2js = require('xml2js');

function getPath(filePath) {
	const pathObj = path.parse(path.join(filePath));
	if (pathObj.root === path.join(filePath)) {
		return '';
	}
	if (!fs.existsSync(path.join(filePath))) {
		return '';
	}
	if (fs.existsSync(path.join(filePath, '.yonstudio', '.project'))) {
		return filePath;
	} else {
		const appInfo = getProjectAppInfo(filePath)
		if (appInfo.appId) {
			writerProject(filePath);
			return filePath;
		} else {
			return getPath(path.resolve(filePath, '..'));
		}
	}
}

function writerProject(project) {
	if (!fs.existsSync(path.resolve(project, '.yonstudio', '.project'))) {
		const appInfo = getProjectAppInfo(project)
		if (appInfo.appId && appInfo.appName) {
			if (!fs.existsSync(path.join(project, '.yonstudio'))) {
				fs.mkdirSync(path.resolve(project, '.yonstudio'));
			}
			const appinfo = {
				appid: appInfo.appId,
				appname: appInfo.appName,
				apptype: 'app'
			}
			if (os.platform() !== 'darwin') {
				fs.writeFileSync(path.resolve(project, '.yonstudio', '.project'), JSON.stringify(appinfo), { encoding: 'utf8' });
				// addToIgnore(path.join(project))
			}
		}
	}
}

function getProjectAppid(project) {
	return getProjectAppInfo(project).appId;
}

async function addToIgnore(localUrl) {
	const dir = path.join(localUrl, '.yonstudio');
	if (fs.existsSync(dir)) {
		let ignoredir = '';
		if (fs.existsSync(path.join(localUrl, '.svn'))) {
			ignoredir = path.join(localUrl, '.svn');
		} else if (fs.existsSync(path.join(localUrl, '.yonstudio'))) {
			ignoredir = path.join(localUrl, '.yonstudio');
		}
		if (ignoredir !== '' && fs.existsSync(path.join(ignoredir, '.svnignore'))) {
			const ignore = fs.readFileSync(path.join(ignoredir, '.svnignore'), { encoding: 'utf8' });
			if (!ignore.includes('.yonstudio')) {
				await vscode.commands.executeCommand('svn.addToIgnoreExplorer', vscode.Uri.file(dir), [vscode.Uri.file(dir)]);
				const data = '.yonstudio' + os.EOL;
				fs.appendFileSync(path.join(ignoredir, '.svnignore'), data, { encoding: 'utf8' });
			}
		}
	}
}

function createAppidFolder(projectPath, fileList) {
	const appid = getProjectAppid(projectPath);
	// const appidFolder = path.join(projectPath, '.bin', appid);
	const appidFolder = path.join(os.tmpdir(), '.yonstudio', appid);
	if (!fs.existsSync(appidFolder)) {
		fs.mkdirSync(appidFolder, { recursive: true });
	}

	fileList.forEach(file => {

		const dest = path.join(appidFolder, file.replace(projectPath, ''))
		if (!fs.existsSync(path.dirname(dest))) {
			fs.mkdirSync(path.dirname(dest), { recursive: true })
		}

		if (file.indexOf('.stml') >= 0) {
			if (file.indexOf(path.join(projectPath, 'pages')) >= 0) {
				const sourceFile = file.replace(path.join(projectPath, 'pages'), path.join(projectPath, '.bin')).replace('.stml', '.js')
				fs.copyFileSync(sourceFile, dest.replace('.stml', '.js'));
			}
		} else {
			if (file.indexOf('.css') >= 0) {
				if (fs.existsSync(file + '.json')) {
					const srcFile = file + '.json';
					const destFile = dest + '.json';
					fs.copyFileSync(srcFile, destFile);
				}
			}
			fs.copyFileSync(file, dest);
		}
	});
}

function getProjectAppInfo(projectPath) {
	let appInfo = {
		appId: '',
		appName: ''
	}
	const configPath = path.join(projectPath, 'config.xml')
	if (fs.existsSync(configPath)) {
		const configContent = fs.readFileSync(configPath, { encoding: 'utf-8' });
		xml2js.parseString(configContent, { trim: true, attrkey: 'nodeAttr', charkey: 'char', explicitArray: false }, (err, result) => {
			if (!err) {
				appInfo['appId'] = result.widget.nodeAttr.id;
				appInfo['appName'] = result.widget.name;
			}
		})
	}

	return appInfo
}

exports.getPath = getPath;
exports.getProjectAppid = getProjectAppid;
exports.createAppidFolder = createAppidFolder;
