"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getProjectPath = exports.requireUncached = exports.findConfigFile = void 0;
const ts = require("typescript");
const path = require("path");
const fs = require("fs");
const os = require("os");
const xml2js = require("xml2js");
function findConfigFile(findPath, configName) {
    return ts.findConfigFile(findPath, ts.sys.fileExists, configName);
}
exports.findConfigFile = findConfigFile;
function requireUncached(module) {
    delete require.cache[require.resolve(module)];
    return require(module);
}
exports.requireUncached = requireUncached;
function getProjectPath(filePath) {
    const pathObj = path.parse(path.join(filePath));
    if (pathObj.root === path.join(filePath)) {
        return '';
    }
    if (!fs.existsSync(path.join(filePath))) {
        return '';
    }
    if (fs.existsSync(path.join(filePath, '.yonstudio', '.project'))) {
        const project = fs.readFileSync(path.join(filePath, '.yonstudio', '.project'), { encoding: 'utf8' });
        const projectObj = JSON.parse(project);
        if (projectObj.apptype === 'app'
            || projectObj.apptype === 'app-mini'
            || projectObj.apptype === 'native-app'
            || projectObj.apptype === 'mx-app'
            || projectObj.apptype === 'cli-app') {
            return filePath;
        }
        else if (projectObj.apptype === 'miniapp') {
            return path.join(path.resolve(filePath, '..'), projectObj.appname);
        }
        else if (projectObj.apptype === 'webapp') {
            return path.join(path.resolve(filePath, '..'), projectObj.appname);
        }
        else if (projectObj.apptype === 'multi_terminal') {
            return path.join(filePath, projectObj.appname);
        }
        else {
            return '';
        }
    }
    else {
        if (fs.existsSync(path.join(filePath, 'config.xml'))) {
            const configContent = fs.readFileSync(path.join(filePath, 'config.xml'), { encoding: 'utf8' });
            const appid = getAppIdByConfig(configContent);
            if (appid) {
                writerProject(filePath, appid);
                return filePath;
            }
            else {
                return getProjectPath(path.resolve(filePath, '..'));
            }
        }
        else {
            return getProjectPath(path.resolve(filePath, '..'));
        }
    }
}
exports.getProjectPath = getProjectPath;
function writerProject(project, appid) {
    if (!fs.existsSync(path.resolve(project, '.yonstudio', '.project'))) {
        if (fs.existsSync(path.resolve(project, 'config.xml'))) {
            const configContent = fs.readFileSync(path.resolve(project, 'config.xml'), { encoding: 'utf8' });
            const appNameInfo = configContent.match(/\<name\>([\s\S]*)\<\/name\>/);
            if (appNameInfo) {
                if (!fs.existsSync(path.join(project, '.yonstudio'))) {
                    fs.mkdirSync(path.resolve(project, '.yonstudio'));
                }
                const appinfo = {
                    appid: appid,
                    appname: appNameInfo[1],
                    apptype: 'app'
                };
                if (os.platform() !== 'darwin') {
                    fs.writeFileSync(path.resolve(project, '.yonstudio', '.project'), JSON.stringify(appinfo), { encoding: 'utf8' });
                }
            }
        }
    }
}
function getAppIdByConfig(config) {
    let appid = '';
    xml2js.parseString(config, { trim: true, attrkey: 'nodeAttr', charkey: 'char', explicitArray: false }, (err, result) => {
        if (!err) {
            appid = result.widget.nodeAttr.id;
        }
    });
    return appid;
}
//# sourceMappingURL=workspace.js.map