"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.modulePathToValidIdentifier = exports.toMarkupContent = exports.indentSection = exports.removeQuotes = exports.getWordAtText = void 0;
const vscode_languageserver_1 = require("vscode-languageserver");
const path_1 = require("path");
function getWordAtText(text, offset, wordDefinition) {
    let lineStart = offset;
    while (lineStart > 0 && !isNewlineCharacter(text.charCodeAt(lineStart - 1))) {
        lineStart--;
    }
    const offsetInLine = offset - lineStart;
    const lineText = text.substr(lineStart);
    // make a copy of the regex as to not keep the state
    const flags = wordDefinition.ignoreCase ? 'gi' : 'g';
    wordDefinition = new RegExp(wordDefinition.source, flags);
    let match = wordDefinition.exec(lineText);
    while (match && match.index + match[0].length < offsetInLine) {
        match = wordDefinition.exec(lineText);
    }
    if (match && match.index <= offsetInLine) {
        return { start: match.index + lineStart, length: match[0].length };
    }
    return { start: offset, length: 0 };
}
exports.getWordAtText = getWordAtText;
function removeQuotes(str) {
    return str.replace(/["']/g, '');
}
exports.removeQuotes = removeQuotes;
const CR = '\r'.charCodeAt(0);
const NL = '\n'.charCodeAt(0);
function isNewlineCharacter(charCode) {
    return charCode === CR || charCode === NL;
}
const nonEmptyLineRE = /^(?!$)/gm;
/**
 *  wrap text in section tags like <template>, <style>
 *  add leading and trailing newline and optional indentation
 */
function indentSection(text, options) {
    const initialIndent = generateIndent(options);
    return text.replace(nonEmptyLineRE, initialIndent);
}
exports.indentSection = indentSection;
function generateIndent(options) {
    if (!options.options.useTabs) {
        return ' '.repeat(options.options.tabSize);
    }
    else {
        return '\t';
    }
}
function toMarkupContent(value) {
    if (!value) {
        return '';
    }
    return typeof value === 'string' ? { kind: vscode_languageserver_1.MarkupKind.Markdown, value } : value;
}
exports.toMarkupContent = toMarkupContent;
// Convert module path to valid typescript identifier
// https://github.com/microsoft/TypeScript/blob/master/src/services/codefixes/importFixes.ts#L951
function modulePathToValidIdentifier(tsModule, modulePath, target) {
    const baseName = path_1.basename(modulePath, '.vue');
    let res = '';
    let lastCharWasValid = true;
    const firstCharCode = baseName.charCodeAt(0);
    if (tsModule.isIdentifierStart(firstCharCode, target)) {
        res += String.fromCharCode(firstCharCode);
    }
    else {
        lastCharWasValid = false;
    }
    for (let i = 1; i < baseName.length; i++) {
        const ch = baseName.charCodeAt(i);
        const isValid = tsModule.isIdentifierPart(ch, target);
        if (isValid) {
            let char = String.fromCharCode(ch);
            if (!lastCharWasValid) {
                char = char.toUpperCase();
            }
            res += char;
        }
        lastCharWasValid = isValid;
    }
    return res;
}
exports.modulePathToValidIdentifier = modulePathToValidIdentifier;
//# sourceMappingURL=strings.js.map