"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VLS = void 0;
const path = require("path");
const paths_1 = require("../utils/paths");
const vscode_languageserver_1 = require("vscode-languageserver");
const vscode_uri_1 = require("vscode-uri");
const nullMode_1 = require("../modes/nullMode");
const dependencyService_1 = require("./dependencyService");
const _ = require("lodash");
const documentService_1 = require("./documentService");
const log_1 = require("../log");
const config_1 = require("../config");
const javascript_1 = require("../modes/script/javascript");
const cancellationToken_1 = require("../utils/cancellationToken");
const workspace_1 = require("../utils/workspace");
const projectService_1 = require("./projectService");
const EnvironmentService_1 = require("./EnvironmentService");
const vueVersion_1 = require("../utils/vueVersion");
const fs_1 = require("fs");
const sleep_1 = require("../utils/sleep");
class VLS {
    constructor(lspConnection) {
        this.lspConnection = lspConnection;
        this.pendingValidationRequests = {};
        this.cancellationTokenValidationRequests = {};
        this.validationDelayMs = 200;
        this.documentService = new documentService_1.DocumentService(this.lspConnection);
        this.workspaces = new Map();
        this.projects = new Map();
        this.nodeModulesMap = new Map();
        this.loadingProjects = [];
        this.globalSnippetDir = '';
    }
    async init(params) {
        var _a, _b, _c, _d;
        const workspaceFolders = Array.isArray(params.workspaceFolders) && ((_a = params.capabilities.workspace) === null || _a === void 0 ? void 0 : _a.workspaceFolders)
            ? params.workspaceFolders.map(el => ({ name: el.name, fsPath: paths_1.getFileFsPath(el.uri) }))
            : params.rootPath
                ? [{ name: '', fsPath: paths_1.normalizeFileNameToFsPath(params.rootPath) }]
                : [];
        if (workspaceFolders.length === 0) {
            console.error('No workspace path found. Vetur initialization failed.');
            // return {
            //   capabilities: {}
            // };
            return;
        }
        this.globalSnippetDir = (_b = params.initializationOptions) === null || _b === void 0 ? void 0 : _b.globalSnippetDir;
        await Promise.all(workspaceFolders.map(workspace => this.addWorkspace(workspace)));
        this.workspaceConfig = this.getVLSFullConfig({}, (_c = params.initializationOptions) === null || _c === void 0 ? void 0 : _c.config);
        if ((_d = params.capabilities.workspace) === null || _d === void 0 ? void 0 : _d.workspaceFolders) {
            this.setupWorkspaceListeners();
        }
        this.setupConfigListeners();
        this.setupLSPHandlers();
        this.setupCustomLSPHandlers();
        this.setupFileChangeListeners();
        this.lspConnection.onShutdown(() => {
            this.dispose();
        });
    }
    listen() {
        this.lspConnection.listen();
    }
    getVLSFullConfig(settings, config) {
        const result = config ? _.merge(config_1.getDefaultVLSConfig(), config) : config_1.getDefaultVLSConfig();
        Object.keys(settings).forEach(key => {
            _.set(result, key, settings[key]);
        });
        return result;
    }
    async addWorkspace(workspace) {
        // Enable Yarn PnP support https://yarnpkg.com/features/pnp
        if (!process.versions.pnp) {
            if (fs_1.existsSync(path.join(workspace.fsPath, '.pnp.js'))) {
                require(path.join(workspace.fsPath, '.pnp.js')).setup();
            }
            else if (fs_1.existsSync(path.join(workspace.fsPath, '.pnp.cjs'))) {
                require(path.join(workspace.fsPath, '.pnp.cjs')).setup();
            }
        }
        const veturConfigPath = workspace_1.findConfigFile(workspace.fsPath, 'vetur.config.js');
        const rootPathForConfig = paths_1.normalizeFileNameToFsPath(veturConfigPath ? path.dirname(veturConfigPath) : workspace.fsPath);
        if (!this.workspaces.has(rootPathForConfig)) {
            this.workspaces.set(rootPathForConfig, {
                name: workspace.name,
                ...(await config_1.getVeturFullConfig(rootPathForConfig, workspace.fsPath, veturConfigPath ? workspace_1.requireUncached(veturConfigPath) : {})),
                isExistVeturConfig: !!veturConfigPath,
                workspaceFsPath: workspace.fsPath
            });
        }
    }
    setupWorkspaceListeners() {
        this.lspConnection.onInitialized(() => {
            this.lspConnection.workspace.onDidChangeWorkspaceFolders(async (e) => {
                await Promise.all(e.added.map(el => this.addWorkspace({ name: el.name, fsPath: paths_1.getFileFsPath(el.uri) })));
            });
        });
    }
    setupConfigListeners() {
        this.lspConnection.onDidChangeConfiguration(async ({ settings }) => {
            var _a, _b, _c, _d, _e, _f;
            this.workspaceConfig = this.getVLSFullConfig({}, settings);
            let isFormatEnable = (_d = (_c = (_b = (_a = this.workspaceConfig) === null || _a === void 0 ? void 0 : _a.stml) === null || _b === void 0 ? void 0 : _b.format) === null || _c === void 0 ? void 0 : _c.enable) !== null && _d !== void 0 ? _d : false;
            log_1.logger.setLevel((_f = (_e = this.workspaceConfig) === null || _e === void 0 ? void 0 : _e.stml) === null || _f === void 0 ? void 0 : _f.dev.logLevel);
            this.projects.forEach(project => {
                const veturConfig = this.workspaces.get(project.env.getRootPathForConfig());
                if (!veturConfig) {
                    return;
                }
                const fullConfig = this.getVLSFullConfig(veturConfig.settings, this.workspaceConfig);
                project.env.configure(fullConfig);
                isFormatEnable = isFormatEnable || fullConfig.stml.format.enable;
            });
            this.setupDynamicFormatters(isFormatEnable);
        });
        this.documentService.getAllDocuments().forEach(this.triggerValidation);
    }
    getAllProjectConfigs() {
        return _.flatten(Array.from(this.workspaces.entries()).map(([rootPathForConfig, veturConfig]) => veturConfig.projects.map(project => ({
            ...project,
            rootPathForConfig,
            vlsFullConfig: this.getVLSFullConfig(veturConfig.settings, this.workspaceConfig),
            workspaceFsPath: veturConfig.workspaceFsPath,
            isExistVeturConfig: veturConfig.isExistVeturConfig
        }))))
            .map(project => ({
            vlsFullConfig: project.vlsFullConfig,
            isExistVeturConfig: project.isExistVeturConfig,
            rootPathForConfig: project.rootPathForConfig,
            workspaceFsPath: project.workspaceFsPath,
            rootFsPath: project.root,
            tsconfigPath: project.tsconfig,
            packagePath: project.package,
            snippetFolder: project.snippetFolder,
            globalComponents: project.globalComponents
        }))
            .sort((a, b) => paths_1.getPathDepth(b.rootFsPath, '/') - paths_1.getPathDepth(a.rootFsPath, '/'));
    }
    warnProjectIfNeed(projectConfig) {
        var _a;
        if (projectConfig.vlsFullConfig.stml.ignoreProjectWarning) {
            return;
        }
        const isFileCanAccess = (fsPath) => {
            try {
                fs_1.accessSync(fsPath, fs_1.constants.R_OK);
                return true;
            }
            catch {
                return false;
            }
        };
        const showErrorIfCantAccess = (name, fsPath) => {
            this.lspConnection.window.showErrorMessage(`Vetur can't access ${fsPath} for ${name}.`);
        };
        // const showWarningAndLearnMore = (message: string, url: string) => {
        //   this.lspConnection.window.showWarningMessage(message, { title: 'Learn More' }).then(action => {
        //     if (action) {
        //       this.openWebsite(url);
        //     }
        //   });
        // };
        // const getCantFindMessage = (fileNames: string[]) =>
        //   `Vetur can't find ${fileNames.map(el => `\`${el}\``).join(' or ')} in ${projectConfig.rootFsPath}.`;
        if (!projectConfig.tsconfigPath) {
            // showWarningAndLearnMore(
            //   getCantFindMessage(['tsconfig.json', 'jsconfig.json']),
            //   'https://vuejs.github.io/vetur/guide/FAQ.html#vetur-can-t-find-tsconfig-json-jsconfig-json-in-xxxx-xxxxxx'
            // );
        }
        else if (!isFileCanAccess(projectConfig.tsconfigPath)) {
            showErrorIfCantAccess('ts/js config', projectConfig.tsconfigPath);
        }
        else {
            if (!projectConfig.isExistVeturConfig &&
                ![
                    paths_1.normalizeFileNameResolve(projectConfig.rootFsPath, 'tsconfig.json'),
                    paths_1.normalizeFileNameResolve(projectConfig.rootFsPath, 'jsconfig.json')
                ].includes((_a = projectConfig.tsconfigPath) !== null && _a !== void 0 ? _a : '')) {
                // showWarningAndLearnMore(
                //   `Vetur find \`tsconfig.json\`/\`jsconfig.json\`, but they aren\'t in the project root.`,
                //   'https://vuejs.github.io/vetur/guide/FAQ.html#vetur-find-xxx-but-they-aren-t-in-the-project-root'
                // );
            }
        }
        if (!projectConfig.packagePath) {
            // showWarningAndLearnMore(
            //   getCantFindMessage(['package.json']),
            //   'https://vuejs.github.io/vetur/guide/FAQ.html#vetur-can-t-find-package-json-in-xxxx-xxxxxx'
            // );
        }
        else if (!isFileCanAccess(projectConfig.packagePath)) {
            showErrorIfCantAccess('ts/js config', projectConfig.packagePath);
        }
        else {
            if (!projectConfig.isExistVeturConfig &&
                paths_1.normalizeFileNameResolve(projectConfig.rootFsPath, 'package.json') !== projectConfig.packagePath) {
                // showWarningAndLearnMore(
                //   `Vetur find \`package.json\`/, but they aren\'t in the project root.`,
                //   'https://vuejs.github.io/vetur/guide/FAQ.html#vetur-find-xxx-but-they-aren-t-in-the-project-root'
                // );
            }
        }
    }
    async getProjectService(uri) {
        var _a;
        const projectConfigs = this.getAllProjectConfigs();
        const docFsPath = paths_1.getFileFsPath(uri);
        const projectConfig = projectConfigs.find(projectConfig => docFsPath.startsWith(projectConfig.rootFsPath));
        if (!projectConfig) {
            return undefined;
        }
        if (this.projects.has(projectConfig.rootFsPath)) {
            return this.projects.get(projectConfig.rootFsPath);
        }
        // Load project once
        if (this.loadingProjects.includes(projectConfig.rootFsPath)) {
            while (!this.projects.has(projectConfig.rootFsPath)) {
                await sleep_1.sleep(500);
            }
            return this.projects.get(projectConfig.rootFsPath);
        }
        // init project
        // Yarn Pnp don't need this. https://yarnpkg.com/features/pnp
        this.loadingProjects.push(projectConfig.rootFsPath);
        const workDoneProgress = await this.lspConnection.window.createWorkDoneProgress();
        workDoneProgress.begin(`Load project: ${projectConfig.rootFsPath}`, undefined);
        const nodeModulePaths = (_a = this.nodeModulesMap.get(projectConfig.rootPathForConfig)) !== null && _a !== void 0 ? _a : dependencyService_1.createNodeModulesPaths(projectConfig.rootPathForConfig);
        if (this.nodeModulesMap.has(projectConfig.rootPathForConfig)) {
            this.nodeModulesMap.set(projectConfig.rootPathForConfig, nodeModulePaths);
        }
        const dependencyService = await dependencyService_1.createDependencyService(projectConfig.rootPathForConfig, projectConfig.workspaceFsPath, projectConfig.vlsFullConfig.stml.useWorkspaceDependencies, nodeModulePaths, projectConfig.vlsFullConfig.typescript.tsdk);
        this.warnProjectIfNeed(projectConfig);
        const project = await projectService_1.createProjectService(EnvironmentService_1.createEnvironmentService(projectConfig.rootPathForConfig, projectConfig.rootFsPath, projectConfig.tsconfigPath, projectConfig.packagePath, projectConfig.snippetFolder, projectConfig.globalComponents, projectConfig.vlsFullConfig), this.documentService, this.globalSnippetDir, dependencyService);
        this.projects.set(projectConfig.rootFsPath, project);
        workDoneProgress.done();
        return project;
    }
    setupLSPHandlers() {
        this.lspConnection.onCompletion(this.onCompletion.bind(this));
        this.lspConnection.onCompletionResolve(this.onCompletionResolve.bind(this));
        this.lspConnection.onDefinition(this.onDefinition.bind(this));
        this.lspConnection.onDocumentFormatting(this.onDocumentFormatting.bind(this));
        this.lspConnection.onDocumentHighlight(this.onDocumentHighlight.bind(this));
        this.lspConnection.onDocumentLinks(this.onDocumentLinks.bind(this));
        this.lspConnection.onDocumentSymbol(this.onDocumentSymbol.bind(this));
        this.lspConnection.onHover(this.onHover.bind(this));
        this.lspConnection.onReferences(this.onReferences.bind(this));
        this.lspConnection.onSignatureHelp(this.onSignatureHelp.bind(this));
        this.lspConnection.onFoldingRanges(this.onFoldingRanges.bind(this));
        this.lspConnection.onCodeAction(this.onCodeAction.bind(this));
        this.lspConnection.onDocumentColor(this.onDocumentColors.bind(this));
        this.lspConnection.onColorPresentation(this.onColorPresentations.bind(this));
        this.lspConnection.onExecuteCommand(this.executeCommand.bind(this));
    }
    setupCustomLSPHandlers() {
        this.lspConnection.onRequest('$/doctor', async ({ fileName }) => {
            var _a, _b;
            const uri = paths_1.getFsPathToUri(fileName);
            const projectConfigs = this.getAllProjectConfigs();
            const project = await this.getProjectService(uri);
            return JSON.stringify({
                name: 'Vetur doctor info',
                fileName,
                currentProject: {
                    vueVersion: project ? vueVersion_1.getVueVersionKey(project === null || project === void 0 ? void 0 : project.env.getVueVersion()) : null,
                    rootPathForConfig: (_a = project === null || project === void 0 ? void 0 : project.env.getRootPathForConfig()) !== null && _a !== void 0 ? _a : null,
                    projectRootFsPath: (_b = project === null || project === void 0 ? void 0 : project.env.getProjectRoot()) !== null && _b !== void 0 ? _b : null
                },
                activeProjects: Array.from(this.projects.keys()),
                projectConfigs
            }, null, 2);
        });
        this.lspConnection.onRequest('$/queryVirtualFileInfo', async ({ fileName, currFileText }) => {
            const project = await this.getProjectService(paths_1.getFsPathToUri(fileName));
            return (project === null || project === void 0 ? void 0 : project.languageModes.getMode('vue-html')).queryVirtualFileInfo(fileName, currFileText);
        });
        this.lspConnection.onRequest('$/getDiagnostics', async (params) => {
            const doc = this.documentService.getDocument(params.uri);
            if (doc) {
                const diagnostics = await this.doValidate(doc);
                return diagnostics !== null && diagnostics !== void 0 ? diagnostics : [];
            }
            return [];
        });
    }
    async setupDynamicFormatters(enable) {
        if (enable) {
            if (!this.documentFormatterRegistration) {
                this.documentFormatterRegistration = await this.lspConnection.client.register(vscode_languageserver_1.DocumentFormattingRequest.type, {
                    documentSelector: [{ language: 'stml' }]
                });
            }
        }
        else {
            if (this.documentFormatterRegistration) {
                this.documentFormatterRegistration.dispose();
            }
        }
    }
    setupFileChangeListeners() {
        this.documentService.onDidChangeContent(change => {
            this.triggerValidation(change.document);
        });
        this.documentService.onDidClose(e => {
            this.removeDocument(e.document);
            this.lspConnection.sendDiagnostics({ uri: e.document.uri, diagnostics: [] });
        });
        this.lspConnection.onDidChangeWatchedFiles(({ changes }) => {
            changes.forEach(async (c) => {
                var _a, _b;
                if (c.type === vscode_languageserver_1.FileChangeType.Changed) {
                    const fsPath = paths_1.getFileFsPath(c.uri);
                    // when `vetur.config.js` changed
                    if (this.workspaces.has(fsPath)) {
                        log_1.logger.logInfo(`refresh vetur config when ${fsPath} changed.`);
                        const name = (_b = (_a = this.workspaces.get(fsPath)) === null || _a === void 0 ? void 0 : _a.name) !== null && _b !== void 0 ? _b : '';
                        this.workspaces.delete(fsPath);
                        await this.addWorkspace({ name, fsPath });
                        this.projects.forEach((project, projectRoot) => {
                            if (project.env.getRootPathForConfig() === fsPath) {
                                project.dispose();
                                this.projects.delete(projectRoot);
                            }
                        });
                        return;
                    }
                    const project = await this.getProjectService(c.uri);
                    project === null || project === void 0 ? void 0 : project.languageModes.getAllModes().forEach(m => {
                        if (m.onDocumentChanged) {
                            m.onDocumentChanged(fsPath);
                        }
                    });
                }
            });
            this.documentService.getAllDocuments().forEach(d => {
                this.triggerValidation(d);
            });
        });
    }
    /**
     * Custom Notifications
     */
    openWebsite(url) {
        this.lspConnection.sendNotification('$/openWebsite', url);
    }
    /**
     * Language Features
     */
    async onDocumentFormatting(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onDocumentFormatting(params)) !== null && _a !== void 0 ? _a : [];
    }
    async onCompletion(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onCompletion(params)) !== null && _a !== void 0 ? _a : nullMode_1.NULL_COMPLETION;
    }
    async onCompletionResolve(item) {
        var _a;
        if (!item.data)
            return item;
        const project = await this.getProjectService(item.data.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onCompletionResolve(item)) !== null && _a !== void 0 ? _a : item;
    }
    async onHover(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onHover(params)) !== null && _a !== void 0 ? _a : nullMode_1.NULL_HOVER;
    }
    async onDocumentHighlight(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onDocumentHighlight(params)) !== null && _a !== void 0 ? _a : [];
    }
    async onDefinition(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onDefinition(params)) !== null && _a !== void 0 ? _a : [];
    }
    async onReferences(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onReferences(params)) !== null && _a !== void 0 ? _a : [];
    }
    async onDocumentLinks(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onDocumentLinks(params)) !== null && _a !== void 0 ? _a : [];
    }
    async onDocumentSymbol(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onDocumentSymbol(params)) !== null && _a !== void 0 ? _a : [];
    }
    async onDocumentColors(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onDocumentColors(params)) !== null && _a !== void 0 ? _a : [];
    }
    async onColorPresentations(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onColorPresentations(params)) !== null && _a !== void 0 ? _a : [];
    }
    async onSignatureHelp(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onSignatureHelp(params)) !== null && _a !== void 0 ? _a : nullMode_1.NULL_SIGNATURE;
    }
    async onFoldingRanges(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onFoldingRanges(params)) !== null && _a !== void 0 ? _a : [];
    }
    async onCodeAction(params) {
        var _a;
        const project = await this.getProjectService(params.textDocument.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.onCodeAction(params)) !== null && _a !== void 0 ? _a : [];
    }
    async getRefactorEdits(refactorAction) {
        var _a;
        const project = await this.getProjectService(vscode_uri_1.URI.file(refactorAction.fileName).toString());
        return (_a = project === null || project === void 0 ? void 0 : project.getRefactorEdits(refactorAction)) !== null && _a !== void 0 ? _a : undefined;
    }
    triggerValidation(textDocument) {
        if (textDocument.uri.includes('node_modules')) {
            return;
        }
        this.cleanPendingValidation(textDocument);
        this.cancelPastValidation(textDocument);
        this.pendingValidationRequests[textDocument.uri] = setTimeout(() => {
            delete this.pendingValidationRequests[textDocument.uri];
            this.cancellationTokenValidationRequests[textDocument.uri] = new cancellationToken_1.VCancellationTokenSource();
            this.validateTextDocument(textDocument, this.cancellationTokenValidationRequests[textDocument.uri].token);
        }, this.validationDelayMs);
    }
    cancelPastValidation(textDocument) {
        const source = this.cancellationTokenValidationRequests[textDocument.uri];
        if (source) {
            source.cancel();
            source.dispose();
            delete this.cancellationTokenValidationRequests[textDocument.uri];
        }
    }
    cleanPendingValidation(textDocument) {
        const request = this.pendingValidationRequests[textDocument.uri];
        if (request) {
            clearTimeout(request);
            delete this.pendingValidationRequests[textDocument.uri];
        }
    }
    async validateTextDocument(textDocument, cancellationToken) {
        const diagnostics = await this.doValidate(textDocument, cancellationToken);
        if (diagnostics) {
            this.lspConnection.sendDiagnostics({ uri: textDocument.uri, diagnostics });
        }
    }
    async doValidate(doc, cancellationToken) {
        var _a;
        const project = await this.getProjectService(doc.uri);
        return (_a = project === null || project === void 0 ? void 0 : project.doValidate(doc, cancellationToken)) !== null && _a !== void 0 ? _a : null;
    }
    async executeCommand(arg) {
        if (arg.command === javascript_1.APPLY_REFACTOR_COMMAND && arg.arguments) {
            const edit = await this.getRefactorEdits(arg.arguments[0]);
            if (edit) {
                // ts-expect-error
                this.lspConnection.sendRequest(vscode_languageserver_1.ApplyWorkspaceEditRequest.type, { edit });
            }
            return;
        }
        log_1.logger.logInfo(`Unknown command ${arg.command}.`);
    }
    async removeDocument(doc) {
        const project = await this.getProjectService(doc.uri);
        project === null || project === void 0 ? void 0 : project.languageModes.onDocumentRemoved(doc);
    }
    dispose() {
        this.projects.forEach(project => {
            project.dispose();
        });
    }
    get capabilities() {
        return {
            textDocumentSync: vscode_languageserver_1.TextDocumentSyncKind.Incremental,
            workspace: { workspaceFolders: { supported: true, changeNotifications: true } },
            completionProvider: { resolveProvider: true, triggerCharacters: ['.', ':', '<', '"', "'", '/', '@', '*', ' '] },
            signatureHelpProvider: { triggerCharacters: ['('] },
            documentFormattingProvider: false,
            hoverProvider: true,
            documentHighlightProvider: true,
            documentLinkProvider: {
                resolveProvider: false
            },
            documentSymbolProvider: true,
            definitionProvider: true,
            referencesProvider: true,
            codeActionProvider: true,
            colorProvider: true,
            executeCommandProvider: {
                commands: [javascript_1.APPLY_REFACTOR_COMMAND]
            },
            foldingRangeProvider: true
        };
    }
}
exports.VLS = VLS;
//# sourceMappingURL=vls.js.map