"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createProjectService = void 0;
const path = require("path");
const vscode_languageserver_1 = require("vscode-languageserver");
const vscode_uri_1 = require("vscode-uri");
const languageModes_1 = require("../embeddedSupport/languageModes");
const nullMode_1 = require("../modes/nullMode");
const paths_1 = require("../utils/paths");
const vueInfoService_1 = require("./vueInfoService");
async function createProjectService(env, documentService, globalSnippetDir, dependencyService) {
    const vueInfoService = new vueInfoService_1.VueInfoService();
    const languageModes = new languageModes_1.LanguageModes();
    function getValidationFlags() {
        const config = env.getConfig();
        return {
            'vue-html': config.stml.validation.template,
            css: config.stml.validation.style,
            postcss: config.stml.validation.style,
            scss: config.stml.validation.style,
            less: config.stml.validation.style,
            javascript: config.stml.validation.script
        };
    }
    vueInfoService.init(languageModes);
    await languageModes.init(env, {
        infoService: vueInfoService,
        dependencyService
    }, globalSnippetDir);
    return {
        env,
        languageModes,
        async onDocumentFormatting({ textDocument, options }) {
            if (!env.getConfig().stml.format.enable) {
                return [];
            }
            const doc = documentService.getDocument(textDocument.uri);
            const modeRanges = languageModes.getAllLanguageModeRangesInDocument(doc);
            const allEdits = [];
            const errMessages = [];
            modeRanges.forEach(modeRange => {
                if (modeRange.mode && modeRange.mode.format) {
                    try {
                        const edits = modeRange.mode.format(doc, { start: modeRange.start, end: modeRange.end }, options);
                        for (const edit of edits) {
                            allEdits.push(edit);
                        }
                    }
                    catch (err) {
                        errMessages.push(err.toString());
                    }
                }
            });
            if (errMessages.length !== 0) {
                console.error('Formatting failed: "' + errMessages.join('\n') + '"');
                return [];
            }
            return allEdits;
        },
        async onCompletion({ textDocument, position, context }) {
            const doc = documentService.getDocument(textDocument.uri);
            const mode = languageModes.getModeAtPosition(doc, position);
            if (mode && mode.doComplete) {
                /**
                 * Only use space as trigger character in `vue-html` mode
                 */
                if (mode.getId() !== 'vue-html' &&
                    context &&
                    (context === null || context === void 0 ? void 0 : context.triggerKind) === vscode_languageserver_1.CompletionTriggerKind.TriggerCharacter &&
                    context.triggerCharacter === ' ') {
                    return nullMode_1.NULL_COMPLETION;
                }
                return mode.doComplete(doc, position);
            }
            return nullMode_1.NULL_COMPLETION;
        },
        async onCompletionResolve(item) {
            if (item.data) {
                const uri = item.data.uri;
                const languageId = item.data.languageId;
                /**
                 * Template files need to go through HTML-template service
                 */
                if (uri.endsWith('.template')) {
                    const doc = documentService.getDocument(uri.slice(0, -'.template'.length));
                    const mode = languageModes.getMode(languageId);
                    if (doc && mode && mode.doResolve) {
                        return mode.doResolve(doc, item);
                    }
                }
                if (uri && languageId) {
                    const doc = documentService.getDocument(uri);
                    const mode = languageModes.getMode(languageId);
                    if (doc && mode && mode.doResolve) {
                        return mode.doResolve(doc, item);
                    }
                }
            }
            return item;
        },
        async onHover({ textDocument, position }) {
            const doc = documentService.getDocument(textDocument.uri);
            const mode = languageModes.getModeAtPosition(doc, position);
            if (mode && mode.doHover) {
                return mode.doHover(doc, position);
            }
            return nullMode_1.NULL_HOVER;
        },
        async onDocumentHighlight({ textDocument, position }) {
            const doc = documentService.getDocument(textDocument.uri);
            const mode = languageModes.getModeAtPosition(doc, position);
            if (mode && mode.findDocumentHighlight) {
                return mode.findDocumentHighlight(doc, position);
            }
            return [];
        },
        async onDefinition({ textDocument, position }) {
            const doc = documentService.getDocument(textDocument.uri);
            const mode = languageModes.getModeAtPosition(doc, position);
            if (mode && mode.findDefinition) {
                return mode.findDefinition(doc, position);
            }
            return [];
        },
        async onReferences({ textDocument, position }) {
            const doc = documentService.getDocument(textDocument.uri);
            const mode = languageModes.getModeAtPosition(doc, position);
            if (mode && mode.findReferences) {
                return mode.findReferences(doc, position);
            }
            return [];
        },
        async onDocumentLinks({ textDocument }) {
            const doc = documentService.getDocument(textDocument.uri);
            const documentContext = {
                resolveReference: ref => {
                    if (ref[0] === '/') {
                        return vscode_uri_1.URI.file(path.resolve(env.getProjectRoot(), ref)).toString();
                    }
                    const fsPath = paths_1.getFileFsPath(doc.uri);
                    return vscode_uri_1.URI.file(path.resolve(fsPath, '..', ref)).toString();
                }
            };
            const links = [];
            languageModes.getAllLanguageModeRangesInDocument(doc).forEach(m => {
                if (m.mode.findDocumentLinks) {
                    links.push.apply(links, m.mode.findDocumentLinks(doc, documentContext));
                }
            });
            return links;
        },
        async onDocumentSymbol({ textDocument }) {
            const doc = documentService.getDocument(textDocument.uri);
            const symbols = [];
            languageModes.getAllLanguageModeRangesInDocument(doc).forEach(m => {
                if (m.mode.findDocumentSymbols) {
                    symbols.push.apply(symbols, m.mode.findDocumentSymbols(doc));
                }
            });
            return symbols;
        },
        async onDocumentColors({ textDocument }) {
            const doc = documentService.getDocument(textDocument.uri);
            const colors = [];
            const distinctModes = new Set();
            languageModes.getAllLanguageModeRangesInDocument(doc).forEach(m => {
                distinctModes.add(m.mode);
            });
            for (const mode of distinctModes) {
                if (mode.findDocumentColors) {
                    colors.push.apply(colors, mode.findDocumentColors(doc));
                }
            }
            return colors;
        },
        async onColorPresentations({ textDocument, color, range }) {
            const doc = documentService.getDocument(textDocument.uri);
            const mode = languageModes.getModeAtPosition(doc, range.start);
            if (mode && mode.getColorPresentations) {
                return mode.getColorPresentations(doc, color, range);
            }
            return [];
        },
        async onSignatureHelp({ textDocument, position }) {
            const doc = documentService.getDocument(textDocument.uri);
            const mode = languageModes.getModeAtPosition(doc, position);
            if (mode && mode.doSignatureHelp) {
                return mode.doSignatureHelp(doc, position);
            }
            return nullMode_1.NULL_SIGNATURE;
        },
        async onFoldingRanges({ textDocument }) {
            const doc = documentService.getDocument(textDocument.uri);
            const lmrs = languageModes.getAllLanguageModeRangesInDocument(doc);
            const result = [];
            lmrs.forEach(lmr => {
                if (lmr.mode.getFoldingRanges) {
                    lmr.mode.getFoldingRanges(doc).forEach(r => result.push(r));
                }
                result.push({
                    startLine: lmr.start.line,
                    startCharacter: lmr.start.character,
                    endLine: lmr.end.line,
                    endCharacter: lmr.end.character
                });
            });
            return result;
        },
        async onCodeAction({ textDocument, range, context }) {
            if (!env.getConfig().stml.languageFeatures.codeActions) {
                return [];
            }
            const doc = documentService.getDocument(textDocument.uri);
            const mode = languageModes.getModeAtPosition(doc, range.start);
            if (languageModes.getModeAtPosition(doc, range.end) !== mode) {
                return [];
            }
            if (mode && mode.getCodeActions) {
                return mode.getCodeActions(doc, range, /*formatParams*/ {}, context);
            }
            return [];
        },
        async doValidate(doc, cancellationToken) {
            const diagnostics = [];
            if (doc.languageId === 'stml') {
                const validationFlags = getValidationFlags();
                for (const lmr of languageModes.getAllLanguageModeRangesInDocument(doc)) {
                    if (lmr.mode.doValidation) {
                        if (validationFlags[lmr.mode.getId()]) {
                            diagnostics.push.apply(diagnostics, await lmr.mode.doValidation(doc, cancellationToken));
                        }
                        // Special case for template type checking
                        else if (lmr.mode.getId() === 'vue-html' &&
                            env.getConfig().stml.experimental.templateInterpolationService) {
                            diagnostics.push.apply(diagnostics, await lmr.mode.doValidation(doc, cancellationToken));
                        }
                    }
                }
            }
            if (cancellationToken === null || cancellationToken === void 0 ? void 0 : cancellationToken.isCancellationRequested) {
                return null;
            }
            return diagnostics;
        },
        async getRefactorEdits(refactorAction) {
            const uri = vscode_uri_1.URI.file(refactorAction.fileName).toString();
            const doc = documentService.getDocument(uri);
            const startPos = doc.positionAt(refactorAction.textRange.pos);
            const mode = languageModes.getModeAtPosition(doc, startPos);
            if (mode && mode.getRefactorEdits) {
                return mode.getRefactorEdits(doc, refactorAction);
            }
            return undefined;
        },
        async dispose() {
            languageModes.dispose();
        }
    };
}
exports.createProjectService = createProjectService;
//# sourceMappingURL=projectService.js.map