"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createDependencyService = exports.createNodeModulesPaths = void 0;
const path = require("path");
const fg = require("fast-glob");
const fs = require("fs");
const util = require("util");
const perf_hooks_1 = require("perf_hooks");
const log_1 = require("../log");
const paths_1 = require("../utils/paths");
// dependencies
const ts = require("typescript");
const prettier = require("prettier");
const prettyHTML = require("@starptech/prettyhtml");
// import * as prettierEslint from 'prettier-eslint';
const prettierTslint = require("prettier-tslint");
const stylusSupremacy = require("stylus-supremacy");
const prettierPluginPug = require("@prettier/plugin-pug");
const readFileAsync = util.promisify(fs.readFile);
const accessFileAsync = util.promisify(fs.access);
function createNodeModulesPaths(rootPath) {
    if (process.versions.pnp) {
        return [];
    }
    const startTime = perf_hooks_1.performance.now();
    const nodeModules = fg.sync('**/node_modules', {
        cwd: rootPath.replace(/\\/g, '/'),
        absolute: true,
        unique: true,
        onlyFiles: false,
        onlyDirectories: true,
        suppressErrors: true,
        deep: 6,
        followSymbolicLinks: true,
        ignore: ['**/node_modules/**/node_modules']
    });
    log_1.logger.logInfo(`Find node_modules paths in ${rootPath} - ${Math.round(perf_hooks_1.performance.now() - startTime)}ms`);
    return nodeModules;
}
exports.createNodeModulesPaths = createNodeModulesPaths;
async function findAllPackages(nodeModulesPaths, moduleName) {
    async function getPackage(nodeModulesPath) {
        const packageJSONPath = path.resolve(nodeModulesPath, moduleName, 'package.json');
        try {
            await accessFileAsync(packageJSONPath, fs.constants.R_OK);
            const info = JSON.parse(await readFileAsync(packageJSONPath, { encoding: 'utf8' }));
            return {
                name: info.name,
                dir: path.dirname(packageJSONPath),
                version: info.version,
                module: require(path.resolve(path.dirname(packageJSONPath), info.main))
            };
        }
        catch {
            return null;
        }
    }
    const packages = (await Promise.all(nodeModulesPaths.map(path => getPackage(path)))).filter(info => info);
    return packages;
}
function compareDependency(a, b) {
    const aDepth = paths_1.getPathDepth(a.dir, path.sep);
    const bDepth = paths_1.getPathDepth(b.dir, path.sep);
    return bDepth - aDepth;
}
const bundledModules = {
    typescript: ts,
    prettier,
    '@starptech/prettyhtml': prettyHTML,
    // 'prettier-eslint': prettierEslint,
    'prettier-tslint': prettierTslint,
    'stylus-supremacy': stylusSupremacy,
    '@prettier/plugin-pug': prettierPluginPug
};
const createDependencyService = async (rootPathForConfig, workspacePath, useWorkspaceDependencies, nodeModulesPaths, tsSDKPath) => {
    let loaded;
    const loadTsSDKPath = () => {
        if (!tsSDKPath) {
            throw new Error('No tsSDKPath in settings');
        }
        const dir = path.isAbsolute(tsSDKPath)
            ? path.resolve(tsSDKPath, '..')
            : path.resolve(workspacePath, tsSDKPath, '..');
        const tsModule = require(dir);
        log_1.logger.logInfo(`Loaded typescript@${tsModule.version} from ${dir} for tsdk.`);
        return {
            dir,
            version: tsModule.version,
            bundled: false,
            module: tsModule
        };
    };
    const loadTypeScript = async () => {
        try {
            if (useWorkspaceDependencies && tsSDKPath) {
                return [loadTsSDKPath()];
            }
            if (useWorkspaceDependencies) {
                const packages = await findAllPackages(nodeModulesPaths, 'typescript');
                if (packages.length === 0) {
                    throw new Error(`No find any packages in ${rootPathForConfig}.`);
                }
                return packages
                    .map(pkg => {
                    log_1.logger.logInfo(`Loaded typescript@${pkg.version} from ${pkg.dir}.`);
                    return {
                        dir: pkg.dir,
                        version: pkg.version,
                        bundled: false,
                        module: pkg.module
                    };
                })
                    .sort(compareDependency);
            }
            throw new Error('No useWorkspaceDependencies.');
        }
        catch (e) {
            log_1.logger.logDebug(e.message);
            log_1.logger.logInfo(`Loaded bundled typescript@${ts.version}.`);
            return [
                {
                    dir: '',
                    version: ts.version,
                    bundled: true,
                    module: ts
                }
            ];
        }
    };
    const loadCommonDep = async (name, bundleModule) => {
        try {
            if (useWorkspaceDependencies) {
                const packages = await findAllPackages(nodeModulesPaths, name);
                if (packages.length === 0) {
                    throw new Error(`No find ${name} packages in ${rootPathForConfig}.`);
                }
                return packages
                    .map(pkg => {
                    log_1.logger.logInfo(`Loaded ${name}@${pkg.version} from ${pkg.dir}.`);
                    return {
                        dir: pkg.dir,
                        version: pkg.version,
                        bundled: false,
                        module: pkg.module
                    };
                })
                    .sort(compareDependency);
            }
            throw new Error('No useWorkspaceDependencies.');
        }
        catch (e) {
            log_1.logger.logDebug(e.message);
            // TODO: Get bundle package version
            log_1.logger.logInfo(`Loaded bundled ${name}.`);
            return [
                {
                    dir: '',
                    version: '',
                    bundled: true,
                    module: bundleModule
                }
            ];
        }
    };
    if (!process.versions.pnp) {
        loaded = {
            typescript: await loadTypeScript(),
            prettier: await loadCommonDep('prettier', bundledModules['prettier']),
            '@starptech/prettyhtml': await loadCommonDep('@starptech/prettyhtml', bundledModules['@starptech/prettyhtml']),
            // 'prettier-eslint': await loadCommonDep('prettier-eslint', bundledModules['prettier-eslint']),
            'prettier-tslint': await loadCommonDep('prettier-tslint', bundledModules['prettier-tslint']),
            'stylus-supremacy': await loadCommonDep('stylus-supremacy', bundledModules['stylus-supremacy']),
            '@prettier/plugin-pug': await loadCommonDep('@prettier/plugin-pug', bundledModules['@prettier/plugin-pug'])
        };
    }
    const get = (lib, filePath) => {
        // We find it when yarn pnp. https://yarnpkg.com/features/pnp
        if (process.versions.pnp) {
            if (!useWorkspaceDependencies) {
                return getBundled(lib);
            }
            if (useWorkspaceDependencies && tsSDKPath && lib === 'typescript') {
                return loadTsSDKPath();
            }
            const pkgPath = require.resolve(lib, { paths: [filePath !== null && filePath !== void 0 ? filePath : workspacePath] });
            return {
                dir: path.dirname(pkgPath),
                version: '',
                bundled: false,
                module: require(pkgPath)
            };
        }
        if (!loaded) {
            throw new Error('Please call init function before get dependency.');
        }
        const deps = loaded[lib];
        // When no filePath, read root workspace dep
        if (!filePath) {
            return deps[deps.length - 1];
        }
        // When only one dep, return it
        if (deps.length === 1) {
            return deps[0];
        }
        const possiblePaths = [];
        let tempPath = path.dirname(filePath);
        while (rootPathForConfig === tempPath ||
            paths_1.getPathDepth(rootPathForConfig, path.sep) > paths_1.getPathDepth(tempPath, path.sep)) {
            possiblePaths.push(path.resolve(tempPath, `node_modules/${lib}`));
            tempPath = path.resolve(tempPath, '../');
        }
        const result = deps.find(dep => possiblePaths.includes(dep.dir));
        return result !== null && result !== void 0 ? result : deps[0];
    };
    const getBundled = (lib) => {
        return {
            dir: '',
            version: '',
            bundled: true,
            module: bundledModules[lib]
        };
    };
    return {
        useWorkspaceDependencies,
        nodeModulesPaths,
        get,
        getBundled
    };
};
exports.createDependencyService = createDependencyService;
//# sourceMappingURL=dependencyService.js.map