"use strict";
/* tslint:disable:max-line-length */
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
/*!
BEGIN THIRD PARTY
*/
/*--------------------------------------------------------------------------------------------
 *  This file is based on or incorporates material from the projects listed below (Third Party IP).
 *  The original copyright notice and the license under which Microsoft received such Third Party IP,
 *  are set forth below. Such licenses and notices are provided for informational purposes only.
 *  Microsoft licenses the Third Party IP to you under the licensing terms for the Microsoft product.
 *  Microsoft reserves all other rights not expressly granted under this agreement, whether by implication,
 *  estoppel or otherwise.
 *--------------------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------------------
 *  Copyright © 2015 W3C® (MIT, ERCIM, Keio, Beihang). This software or document includes includes material copied
 *  from or derived from HTML 5.1 W3C Working Draft (http://www.w3.org/TR/2015/WD-html51-20151008/.)"
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.getHTML5TagProvider = exports.HTML_TAGS = exports.isVoidElement = exports.VOID_ELEMENTS = void 0;
const common_1 = require("./common");
// As defined in https://www.w3.org/TR/html5/syntax.html#void-elements
exports.VOID_ELEMENTS = [
    'area',
    'base',
    'br',
    'col',
    'embed',
    'hr',
    'img',
    'input',
    'keygen',
    'link',
    'menuitem',
    'meta',
    'param',
    'source',
    'track',
    'wbr'
];
function isVoidElement(e) {
    return !!e && binarySearch(exports.VOID_ELEMENTS, e.toLowerCase(), (s1, s2) => s1.localeCompare(s2)) >= 0;
}
exports.isVoidElement = isVoidElement;
function genAttr(attrString) {
    const [label, type] = attrString.split(':');
    return { label, type };
}
function genTag(documentation, attrs) {
    const attributes = attrs && attrs.map(genAttr);
    return new common_1.HTMLTagSpecification(documentation, attributes);
}
// HTML tag information sourced from http://www.w3.org/TR/2015/WD-html51-20151008/
exports.HTML_TAGS = {
    div: genTag('The div element has no special meaning at all. It represents its children. It can be used with the class, lang, and title attributes to mark up semantics common to a group of consecutive elements.'),
    view: genTag('通用容器组件，内部可以放置任意组件。默认布局方式为flex布局. 同div'),
    img: genTag('An img element represents an image.', [
        'src',
        'placeholder',
        'policy:imgp',
        'thumbnail:imgt',
        'mode:imgmode',
        'onerror',
        'onload'
    ]),
    image: genTag('用于显示图片. 同img', [
        'src',
        'placeholder',
        'policy:imgp',
        'thumbnail:imgt',
        'mode:imgmode',
        'onerror',
        'onload'
    ]),
    'movable-area': genTag('movable-view 的可移动区域', [
        'scale-area:b'
    ]),
    'movable-view': genTag('可移动的视图容器，在页面中可以拖拽滑动。movable-view 必须在 movable-area 组件中，并且必须是直接子节点，否则不能移动', [
        'direction',
        'inertia:b',
        'out-of-bounds:b',
        'x',
        'y',
        'damping',
        'friction',
        'disabled:b',
        'scale:b',
        'scale-min',
        'scale-max',
        'scale-value',
        'animation:b',
        'onchange',
        'onscale',
        'onhtouchmove',
        'onvtouchmove'
    ]),
    audio: genTag('用于播放音频', [
        'src',
        'loop:b',
        'controls:b',
        'onerror',
        'onplay',
        'onpause',
        'ontimeupdate',
        'onended'
    ]),
    video: genTag('A video element is used for playing videos or movies, and audio files with captions.', [
        'src',
        'poster',
        'autoplay:b',
        'loop:b',
        'muted:b',
        'controls:b',
        'duration',
        'initial-time',
        'currentTime',
        'playbackRate',
        'onplay',
        'onpause',
        'onended',
        'ontimeupdate',
        'onwaiting',
        'onerror',
        'onprogress',
        'onloadedmetadata'
    ]),
    // Forms
    form: genTag('The form element represents a collection of form-associated elements, some of which can represent editable values that can be submitted to a server for processing.', ['onsubmit', 'onreset']),
    label: genTag("The label element represents a caption in a user interface. The caption can be associated with a specific form control, known as the label element's labeled control, either using the for attribute, or by putting the form control inside the label element itself.", ['for']),
    input: genTag('The input element represents a typed data field, usually with a form control to allow the user to edit the data.', [
        'disabled:b',
        'maxlength',
        'placeholder',
        'type:inputtype',
        'value',
        'autofocus:b',
        'keyboard-type:keyboardtype',
        'confirm-type:confirmtype',
        'confirm-hold:b',
        'oninput',
        'onfocus',
        'onblur',
        'onconfirm'
    ]),
    button: genTag('The button element represents a button labeled by its contents.', [
        'disabled:v',
        'type:bt',
    ]),
    textarea: genTag("The textarea element represents a multiline plain text edit control for the element's raw value. The contents of the control represent the control's default value.", [
        'disabled:v',
        'maxlength',
        'placeholder',
        'value',
        'autofocus:b',
        'auto-height:b',
        'keyboard-type:keyboardtype',
        'confirm-type:confirmtype',
        'oninput',
        'onfocus',
        'onblur',
        'onconfirm',
        'onlinechange'
    ]),
    progress: genTag('The progress element represents the completion progress of a task. The progress is either indeterminate, indicating that progress is being made but that it is not clear how much more work remains to be done before the task is complete (e.g. because the task is waiting for a remote host to respond), or the progress is a number in the range zero to a maximum, giving the fraction of work that has so far been completed.', [
        'value',
        'max',
        'percent',
        'activeColor',
        'backgroundColor',
        'active',
        'active-mode:am',
        'onactiveend'
    ]),
    text: genTag('显示文本信息'),
    switch: genTag('开关选择器', ['checked:b', 'disabled:b', 'color', 'onchange']),
    slider: genTag('滑动选择器', ['min', 'max', 'value', 'step', 'disabled:b', 'backgroundColor', 'activeColor', 'block-size', 'block-color', 'onchange', 'onchanging']),
    picker: genTag('滚动选择器', ['mode:pm', 'disabled', 'oncancel']),
    checkbox: genTag('复选框', ['value', 'checked:b', 'disabled:b', 'color', 'icon', 'selectedIcon', 'onchange']),
    'checkbox-group': genTag('多选框组。内部包含多个 checkbox', ['onchange']),
    radio: genTag('单选框', ['value', 'checked:b', 'disabled:b', 'color', 'icon', 'selectedIcon', 'onchange']),
    'radio-group': genTag('单选框组。内部包含多个radio', ['onchange']),
    camera: genTag('系统相机、扫描二维码', ['mode:cameramode', 'devicePosition:deviceposition', 'flash:cameraflash', 'onstop', 'onerror', 'oninitdone', 'onscancode']),
    swiper: genTag('滑动视图，支持上下或左右滑动。其中只可放置 swiper-item 组件', [
        'indicator-dots:b',
        'indicator-color',
        'indicator-active-color',
        'autoplay:b',
        'current',
        'interval',
        'circular:b',
        'vertical:b',
        'onchange'
    ]),
    'swiper-item': genTag('仅可放置在 swiper 组件中，宽高自动设置为100%'),
    'scroll-view': genTag('滚动视图', [
        'scroll-x:b',
        'scroll-y:b',
        'show-scrollbar:b',
        'bounces:b',
        'paging-enabled:b',
        'scroll-enabled	:b',
        'enable-back-to-top:b',
        'scroll-top',
        'scroll-left',
        'scroll-into-view',
        'scroll-with-animation:b',
        'refresher-enabled:b',
        'refresher-threshold',
        'refresher-background',
        'refresher-triggered:b',
        'upper-threshold',
        'lower-threshold',
        'onscrolltoupper',
        'onscrolltolower',
        'onscroll',
        'onrefresherrefresh',
        'onrefresherrestore'
    ]),
    'list-view': genTag('list-view 定义可复用内容的竖向滚动视图，可以优化内存占用和渲染性能，支持下拉刷新。list-view 里面可放置cell、list-header、list-footer组件，使用 cell 组件作为每项显示内容', [
        'scroll-x:b',
        'scroll-y:b',
        'show-scrollbar:b',
        'bounces:b',
        'paging-enabled:b',
        'scroll-enabled	:b',
        'enable-back-to-top:b',
        'scroll-top',
        'scroll-left',
        'scroll-into-view',
        'scroll-with-animation:b',
        'refresher-enabled:b',
        'refresher-threshold',
        'refresher-background',
        'refresher-triggered:b',
        'upper-threshold',
        'lower-threshold',
        'onscrolltoupper',
        'onscrolltolower',
        'onscroll',
        'onrefresherrefresh',
        'onrefresherrestore',
        'cellTypeKey'
    ]),
    'list-header': genTag('list-header 定义 list-view 的头部. 该组件只能放在 list-view 组件中，内部可放置任意子组件，宽度自动设置为100%'),
    'list-footer': genTag('list-footer 定义 list-view 的底部. 该组件只能放在 list-view 组件中，内部可放置任意子组件，宽度自动设置为100%'),
    cell: genTag('cell 定义 list-view 中的每一项内容。该组件只能放置在 list-view 组件中，内部可放置任意子组件，宽度自动设置为100%', ['cellType']),
    frame: genTag('frame 组件用于显示一个frame，效果同 openFrame 方法一致', [
        'name',
        'url',
        'data',
        'headers',
        'avm:b',
        'useWKWebView:b',
        'historyGestureEnabled:b',
        'pageParam',
        'bounces:b',
        'scrollToTop:b',
        'scrollEnabled:b',
        'vScrollBarEnabled:b',
        'hScrollBarEnabled:b',
        'scaleEnabled:b',
        'progress',
        'reload:b',
        'allowEdit:b',
        'softInputMode:softInputMode',
        'softInputDismissMode',
        'softInputBarEnabled:b',
        'overScrollMode:overScrollMode',
        'dragAndDrop:b',
        'customRefreshHeader'
    ]),
    'frame-group': genTag('frame-group 组件用于显示一个 frame 组，效果同 openFrameGroup 方法一致', [
        'scrollEnabled:b',
        'index',
        'preload',
        'onchange'
    ]),
    refresh: genTag('refresh 为滚动视图中的刷新组件.该组件只能放置在 scroll-view、list-view 等列表组件中，宽度自动设置为100%。内部可放置任意子组件，来实现自定义的下拉刷新、上拉加载更多等功能', [
        'type:refreshtype',
        'threshold',
        'state:refreshstate',
        'onstatechange'
    ]),
    'safe-area': genTag('safe-area 定义”安全“区域. 将该组件放置在页面顶部或底部，可以自动预留出状态栏、底部虚拟Home键等区域，从而使页面其它内容不被遮挡'),
    'rich-text': genTag('rich-text组件用于显示富文本', ['nodes']),
    'grid-view': genTag('grid-view 定义可复用内容的网格滚动视图，支持横向或竖向滚动，可以优化内存占用和渲染性能，支持下拉刷新和上拉加载。可使用 scroll-view 的基本属性.', [
        'column-count',
        'column-width',
        'vertical-spacing',
        'horizontal-spacing',
        'stretch-mode:gridmode',
        'aspect-ratio',
        'inset',
        'cell-type-key',
        'scroll-x:b',
        'scroll-y:b',
        'show-scrollbar:b',
        'bounces:b',
        'paging-enabled:b',
        'scroll-enabled	:b',
        'enable-back-to-top:b',
        'scroll-top',
        'scroll-left',
        'scroll-into-view',
        'scroll-with-animation:b',
        'refresher-enabled:b',
        'refresher-threshold',
        'refresher-background',
        'refresher-triggered:b',
        'upper-threshold',
        'lower-threshold',
        'onscrolltoupper',
        'onscrolltolower',
        'onscroll',
        'onrefresherrefresh',
        'onrefresherrestore'
    ]),
    'dialog': genTag('dialog 定义了一组不参与当前视图渲染的组件，通过 API 触发在当前窗口弹出全局的模态或非模态对话框，对话框内容通过布局自定义实现', [
        'open',
        'cancelable',
        'returnValue',
        'oncancel',
        'onclose'
    ]),
    'sticky': genTag('sticky 为粘性布局组件，与 CSS 中 position: sticky 属性实现的效果一致，当组件在屏幕范围内时，会按照正常的布局排列，当组件滚出屏幕范围时，始终会固定在屏幕顶部', [
        'offset-top',
        'container-id',
        'onscroll'
    ])
};
function getHTML5TagProvider() {
    const globalAttributes = [
        'class',
        'data-',
        'hidden:b',
        'id',
        'style'
    ].map(genAttr);
    const eventHandlers = [
        'abort',
        'blur',
        'canplay',
        'canplaythrough',
        'change',
        'click',
        'contextmenu',
        'dblclick',
        'drag',
        'dragend',
        'dragenter',
        'dragleave',
        'dragover',
        'dragstart',
        'drop',
        'durationchange',
        'emptied',
        'ended',
        'error',
        'focus',
        'formchange',
        'forminput',
        'input',
        'invalid',
        'keydown',
        'keypress',
        'keyup',
        'load',
        'loadeddata',
        'loadedmetadata',
        'loadstart',
        'mousedown',
        'mousemove',
        'mouseout',
        'mouseover',
        'mouseenter',
        'mouseleave',
        'mouseup',
        'mousewheel',
        'pause',
        'play',
        'playing',
        'progress',
        'ratechange',
        'reset',
        'resize',
        'readystatechange',
        'scroll',
        'seeked',
        'seeking',
        'select',
        'show',
        'stalled',
        'submit',
        'suspend',
        'timeupdate',
        'volumechange',
        'waiting'
    ];
    const valueSets = {
        b: ['true', 'false'],
        u: ['true', 'false', 'undefined'],
        o: ['on', 'off'],
        y: ['yes', 'no'],
        w: ['soft', 'hard'],
        d: ['ltr', 'rtl', 'auto'],
        m: ['GET', 'POST', 'dialog'],
        fm: ['GET', 'POST'],
        s: ['row', 'col', 'rowgroup', 'colgroup'],
        am: ['backwards', 'forwards'],
        t: [
            'hidden',
            'text',
            'search',
            'tel',
            'url',
            'email',
            'password',
            'datetime',
            'date',
            'month',
            'week',
            'time',
            'datetime-local',
            'number',
            'range',
            'color',
            'checkbox',
            'radio',
            'file',
            'submit',
            'image',
            'reset',
            'button'
        ],
        inputtype: [
            'text',
            'password'
        ],
        keyboardtype: ['default', 'number', 'decimal', 'tel', 'email', 'url'],
        confirmtype: ['default', 'send', 'search', 'next', 'go', 'done'],
        im: [
            'verbatim',
            'latin',
            'latin-name',
            'latin-prose',
            'full-width-latin',
            'kana',
            'kana-name',
            'katakana',
            'numeric',
            'tel',
            'email',
            'url'
        ],
        imgp: ['default', 'cache_else_network', 'no_cache', 'cache_only'],
        imgmode: [
            'scaleToFill',
            'aspectFit',
            'aspectFill',
            'widthFix',
            'top',
            'bottom',
            'center',
            'left',
            'right',
            'top left',
            'top right',
            'bottom left',
            'bottom right'
        ],
        imgt: ['true', 'false'],
        bt: ['button', 'submit', 'reset'],
        lt: ['1', 'a', 'A', 'i', 'I'],
        mt: ['context', 'toolbar'],
        mit: ['command', 'checkbox', 'radio'],
        et: ['application/x-www-form-urlencoded', 'multipart/form-data', 'text/plain'],
        tk: ['subtitles', 'captions', 'descriptions', 'chapters', 'metadata'],
        pl: ['none', 'metadata', 'auto'],
        sh: ['circle', 'default', 'poly', 'rect'],
        xo: ['anonymous', 'use-credentials'],
        sb: [
            'allow-forms',
            'allow-modals',
            'allow-pointer-lock',
            'allow-popups',
            'allow-popups-to-escape-sandbox',
            'allow-same-origin',
            'allow-scripts',
            'allow-top-navigation'
        ],
        tristate: ['true', 'false', 'mixed', 'undefined'],
        inputautocomplete: [
            'additional-name',
            'address-level1',
            'address-level2',
            'address-level3',
            'address-level4',
            'address-line1',
            'address-line2',
            'address-line3',
            'bday',
            'bday-year',
            'bday-day',
            'bday-month',
            'billing',
            'cc-additional-name',
            'cc-csc',
            'cc-exp',
            'cc-exp-month',
            'cc-exp-year',
            'cc-family-name',
            'cc-given-name',
            'cc-name',
            'cc-number',
            'cc-type',
            'country',
            'country-name',
            'current-password',
            'email',
            'family-name',
            'fax',
            'given-name',
            'home',
            'honorific-prefix',
            'honorific-suffix',
            'impp',
            'language',
            'mobile',
            'name',
            'new-password',
            'nickname',
            'organization',
            'organization-title',
            'pager',
            'photo',
            'postal-code',
            'sex',
            'shipping',
            'street-address',
            'tel-area-code',
            'tel',
            'tel-country-code',
            'tel-extension',
            'tel-local',
            'tel-local-prefix',
            'tel-local-suffix',
            'tel-national',
            'transaction-amount',
            'transaction-currency',
            'url',
            'username',
            'work'
        ],
        autocomplete: ['inline', 'list', 'both', 'none'],
        current: ['page', 'step', 'location', 'date', 'time', 'true', 'false'],
        dropeffect: ['copy', 'move', 'link', 'execute', 'popup', 'none'],
        invalid: ['grammar', 'false', 'spelling', 'true'],
        live: ['off', 'polite', 'assertive'],
        orientation: ['vertical', 'horizontal', 'undefined'],
        relevant: ['additions', 'removals', 'text', 'all', 'additions text'],
        sort: ['ascending', 'descending', 'none', 'other'],
        roles: [
            'alert',
            'alertdialog',
            'button',
            'checkbox',
            'dialog',
            'gridcell',
            'link',
            'log',
            'marquee',
            'menuitem',
            'menuitemcheckbox',
            'menuitemradio',
            'option',
            'progressbar',
            'radio',
            'scrollbar',
            'searchbox',
            'slider',
            'spinbutton',
            'status',
            'switch',
            'tab',
            'tabpanel',
            'textbox',
            'timer',
            'tooltip',
            'treeitem',
            'combobox',
            'grid',
            'listbox',
            'menu',
            'menubar',
            'radiogroup',
            'tablist',
            'tree',
            'treegrid',
            'application',
            'article',
            'cell',
            'columnheader',
            'definition',
            'directory',
            'document',
            'feed',
            'figure',
            'group',
            'heading',
            'img',
            'list',
            'listitem',
            'math',
            'none',
            'note',
            'presentation',
            'region',
            'row',
            'rowgroup',
            'rowheader',
            'separator',
            'table',
            'term',
            'text',
            'toolbar',
            'banner',
            'complementary',
            'contentinfo',
            'form',
            'main',
            'navigation',
            'region',
            'search'
        ],
        cameramode: ['normal', 'scanCode'],
        deviceposition: ['front', 'back'],
        cameraflash: ['auto', 'on', 'off'],
        softInputMode: ['auto', 'resize', 'pan'],
        overScrollMode: ['never', 'always', 'scrolls'],
        refreshtype: ['header', 'footer'],
        refreshstate: ['normal', 'dragging', 'refreshing'],
        gridmode: ['none', 'columnWidth', 'spacingWidth']
    };
    return {
        getId: () => 'html5',
        collectTags: (collector) => common_1.collectTagsDefault(collector, exports.HTML_TAGS),
        collectAttributes: (tag, collector) => {
            common_1.collectAttributesDefault(tag, collector, exports.HTML_TAGS, globalAttributes);
            eventHandlers.forEach(handler => {
                collector(handler, 'event');
            });
        },
        priority: common_1.Priority.Platform,
        collectValues: (tag, attribute, collector) => common_1.collectValuesDefault(tag, attribute, collector, exports.HTML_TAGS, globalAttributes, valueSets)
    };
}
exports.getHTML5TagProvider = getHTML5TagProvider;
/*!
END THIRD PARTY
*/
function binarySearch(array, key, comparator) {
    let low = 0, high = array.length - 1;
    while (low <= high) {
        const mid = ((low + high) / 2) | 0;
        const comp = comparator(array[mid], key);
        if (comp < 0) {
            low = mid + 1;
        }
        else if (comp > 0) {
            high = mid - 1;
        }
        else {
            return mid;
        }
    }
    return -(low + 1);
}
//# sourceMappingURL=htmlTags.js.map