"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.htmlFormat = void 0;
const vscode_languageserver_types_1 = require("vscode-languageserver-types");
const js_beautify_1 = require("js-beautify");
const paths_1 = require("../../../utils/paths");
const prettier_1 = require("../../../utils/prettier");
const utils_1 = require("../stml-parse/utils");
const TEMPLATE_HEAD = '<template>';
const TEMPLATE_TAIL = '</template>';
function htmlFormat(dependencyService, document, currRange, vlsFormatConfig) {
    if (vlsFormatConfig.defaultFormatter.html === 'none') {
        return [];
    }
    const { value, range } = getValueAndRange(document, currRange);
    let originalSource = TEMPLATE_HEAD + value + TEMPLATE_TAIL;
    let beautifiedHtml;
    if (vlsFormatConfig.defaultFormatter.html === 'prettyhtml') {
        beautifiedHtml = formatWithPrettyHtml(dependencyService, paths_1.getFileFsPath(document.uri), originalSource, vlsFormatConfig);
    }
    else if (vlsFormatConfig.defaultFormatter.html === 'prettier') {
        let requirePragma = false;
        // if (value.includes('@')) {
        //   requirePragma = true;
        //   return []
        // }
        originalSource = utils_1.convertJSX(originalSource);
        const prettierResult = formatWithPrettier(dependencyService, originalSource, paths_1.getFileFsPath(document.uri), currRange, vlsFormatConfig, false, requirePragma);
        if (prettierResult[0] && prettierResult[0].newText) {
            beautifiedHtml = prettierResult[0].newText.trim();
        }
        else {
            beautifiedHtml = originalSource;
        }
        // range: Range.create(Position.create(range.start.line, 0), Position.create(range.end.line + 1, range.end.character)),
        return [
            {
                range: vscode_languageserver_types_1.Range.create(vscode_languageserver_types_1.Position.create(range.start.line, 0), vscode_languageserver_types_1.Position.create(range.end.line + 1, range.end.character)),
                newText: beautifiedHtml + '\n'
            }
        ];
    }
    else {
        const newRange = vscode_languageserver_types_1.Range.create(vscode_languageserver_types_1.Position.create(0, 0), vscode_languageserver_types_1.Position.create(range.end.line, range.end.character));
        const htmlResult = formatWithJsBeautify_html(document, newRange, defaultHtmlOptions2);
        if (htmlResult[0] && htmlResult[0].newText) {
            const wrappedHtml = htmlResult[0].newText + '\n';
            return [
                {
                    range: newRange,
                    newText: wrappedHtml
                }
            ];
        }
        else {
            return [];
        }
    }
    const wrappedHtml = beautifiedHtml.substring(TEMPLATE_HEAD.length, beautifiedHtml.length - TEMPLATE_TAIL.length);
    return [
        {
            range,
            newText: wrappedHtml
        }
    ];
}
exports.htmlFormat = htmlFormat;
function formatWithJsBeautify_html(document, range, htmlOptions) {
    var value = document.getText();
    // var includesEnd = true;
    var initialIndentLevel = 0;
    // var tabSize = 4;
    if (range) {
        var startOffset = document.offsetAt(range.start);
        // include all leading whitespace iff at the beginning of the line
        var extendedStart = startOffset;
        while (extendedStart > 0 && isWhitespace(value, extendedStart - 1)) {
            extendedStart--;
        }
        if (extendedStart === 0 || isEOL(value, extendedStart - 1)) {
            startOffset = extendedStart;
        }
        else {
            // else keep at least one whitespace
            if (extendedStart < startOffset) {
                startOffset = extendedStart + 1;
            }
        }
        // include all following whitespace until the end of the line
        var endOffset = document.offsetAt(range.end);
        var extendedEnd = endOffset;
        while (extendedEnd < value.length && isWhitespace(value, extendedEnd)) {
            extendedEnd++;
        }
        if (extendedEnd === value.length || isEOL(value, extendedEnd)) {
            endOffset = extendedEnd;
        }
        range = vscode_languageserver_types_1.Range.create(document.positionAt(startOffset), document.positionAt(endOffset));
        // Do not modify if substring starts in inside an element
        // Ending inside an element is fine as it doesn't cause formatting errors
        var firstHalf = value.substring(0, startOffset);
        if (new RegExp(/.*[<][^>]*$/).test(firstHalf)) {
            //return without modification
            value = value.substring(startOffset, endOffset);
            return [{
                    range: range,
                    newText: value
                }];
        }
        // includesEnd = endOffset === value.length;
        value = value.substring(startOffset, endOffset);
        if (startOffset !== 0) {
            var startOfLineOffset = document.offsetAt(vscode_languageserver_types_1.Position.create(range.start.line, 0));
            initialIndentLevel = computeIndentLevel(document.getText(), startOfLineOffset);
        }
    }
    else {
        range = vscode_languageserver_types_1.Range.create(vscode_languageserver_types_1.Position.create(0, 0), document.positionAt(value.length));
    }
    // value = convertJSX(value);
    // {wrap_attributes:'preserve'} htmlOptions
    var result = js_beautify_1.html_beautify(trimLeft(value), htmlOptions);
    // result = convertHTMLComment(result);
    if (initialIndentLevel > 0) {
        var indent = repeat('\t', initialIndentLevel);
        result = result.split('\n').join('\n' + indent);
        if (range.start.character === 0) {
            result = indent + result; // keep the indent
        }
    }
    return [{
            range: range,
            newText: result
        }];
}
function repeat(value, count) {
    var s = '';
    while (count > 0) {
        if ((count & 1) === 1) {
            s += value;
        }
        value += value;
        count = count >>> 1;
    }
    return s;
}
function isWhitespace(text, offset) {
    return ' \t'.indexOf(text.charAt(offset)) !== -1;
}
function isEOL(text, offset) {
    return '\r\n'.indexOf(text.charAt(offset)) !== -1;
}
function computeIndentLevel(content, offset) {
    var i = offset;
    var nChars = 0;
    var tabSize = 4;
    while (i < content.length) {
        var ch = content.charAt(i);
        if (ch === ' ') {
            nChars++;
        }
        else if (ch === '\t') {
            nChars += tabSize;
        }
        else {
            break;
        }
        i++;
    }
    return Math.floor(nChars / tabSize);
}
function trimLeft(str) {
    return str.replace(/^\s+/, '');
}
function formatWithPrettyHtml(dependencyService, fileFsPath, input, vlsFormatConfig) {
    const prettier = dependencyService.get('prettier', fileFsPath).module;
    const prettierrcOptions = prettier.resolveConfig.sync(fileFsPath, { useCache: false }) || null;
    const prettyhtml = dependencyService.get('@starptech/prettyhtml', fileFsPath).module;
    const result = prettyhtml(input, getPrettyHtmlOptions(prettierrcOptions, vlsFormatConfig));
    return result.contents.trim();
}
// function formatWithJsBeautify(input: string, vlsFormatConfig: VLSFormatConfig): string {
//   const htmlFormattingOptions = _.assign(
//     defaultHtmlOptions1,
//     {
//       indent_with_tabs: vlsFormatConfig.options.useTabs,
//       indent_size: vlsFormatConfig.options.tabSize
//     },
//     vlsFormatConfig.defaultFormatterOptions['js-beautify-html'],
//     { end_with_newline: false }
//   );
//   return htmlBeautify(input, defaultHtmlOptions1);
// }
function formatWithPrettier(dependencyService, code, fileFsPath, range, vlsFormatConfig, initialIndent, requirePragma) {
    const textEdits = prettier_1.prettierify(dependencyService, code, fileFsPath, range, vlsFormatConfig, 'babel', initialIndent, requirePragma);
    return textEdits;
}
function getPrettyHtmlOptions(prettierrcOptions, vlsFormatConfig) {
    const fromVls = {
        useTabs: vlsFormatConfig.options.useTabs,
        tabWidth: vlsFormatConfig.options.tabSize
    };
    const fromPrettier = {};
    if (prettierrcOptions) {
        fromPrettier.useTabs = prettierrcOptions.useTabs;
        fromPrettier.tabWidth = prettierrcOptions.tabWidth;
        fromPrettier.printWidth = prettierrcOptions.printWidth;
    }
    return {
        ...fromVls,
        ...fromPrettier,
        usePrettier: true,
        prettier: {
            ...prettierrcOptions
        },
        ...vlsFormatConfig.defaultFormatterOptions['prettyhtml']
    };
}
function getValueAndRange(document, currRange) {
    let value = document.getText();
    let range = currRange;
    if (currRange) {
        const startOffset = document.offsetAt(currRange.start);
        const endOffset = document.offsetAt(currRange.end);
        value = value.substring(startOffset, endOffset);
    }
    else {
        range = vscode_languageserver_types_1.Range.create(vscode_languageserver_types_1.Position.create(0, 0), document.positionAt(value.length));
    }
    return { value, range };
}
// const defaultHtmlOptions: HTMLBeautifyOptions = {
//   end_with_newline: false, // End output with newline
//   indent_char: ' ', // Indentation character
//   indent_handlebars: false, // e.g. {{#foo}}, {{/foo}}
//   indent_inner_html: false, // Indent <head> and <body> sections
//   indent_scripts: 'keep', // [keep|separate|normal]
//   indent_size: 2, // Indentation size
//   indent_with_tabs: false,
//   max_preserve_newlines: 5, // Maximum number of line breaks to be preserved in one chunk (0 disables)
//   preserve_newlines: true, // Whether existing line breaks before elements should be preserved
//   unformatted: [], // Tags that shouldn't be formatted. Causes mis-alignment
//   wrap_line_length: 0, // Lines should wrap at next opportunity after this number of characters (0 disables)
//   wrap_attributes: 'force-expand-multiline' as any
//   // Wrap attributes to new lines [auto|force|force-aligned|force-expand-multiline] ["auto"]
// };
// const defaultHtmlOptions1: HTMLBeautifyOptions = {
//   indent_size: 4,
//   indent_char: " ",
//   max_preserve_newlines: 5,
//   preserve_newlines: true,
//   indent_scripts: "keep",
//   end_with_newline: false,
//   wrap_line_length: 160,
//   indent_inner_html: false,
//   indent_empty_lines: false,
//   indent_handlebars: false,
//   indent_with_tabs: false,
//   unformatted: []
// };
const defaultHtmlOptions2 = {
    indent_size: 4,
    indent_char: '\t',
    indent_empty_lines: false,
    wrap_line_length: 120,
    unformatted: ['wbr'],
    content_unformatted: ["pre", "code", "textarea", "script"],
    indent_inner_html: false,
    preserve_newlines: true,
    max_preserve_newlines: undefined,
    indent_handlebars: false,
    end_with_newline: false,
    extra_liners: ['head', 'body', '/html'],
    wrap_attributes: 'auto',
    wrap_attributes_indent_size: undefined,
    eol: '\n',
    indent_scripts: 'normal',
    templating: ['none'],
    unformatted_content_delimiter: ''
};
//# sourceMappingURL=htmlFormat.js.map