"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.doComplete = void 0;
const vscode_languageserver_types_1 = require("vscode-languageserver-types");
const htmlScanner_1 = require("../parser/htmlScanner");
const emmet = require("vscode-emmet-helper");
const nullMode_1 = require("../../nullMode");
const modifierProvider_1 = require("../modifierProvider");
const strings_1 = require("../../../utils/strings");
const common_1 = require("../tagProviders/common");
const workspace_1 = require("../../../utils/workspace");
const vscode_uri_1 = require("vscode-uri");
const fs = require("fs");
const path = require("path");
const os = require("os");
function doComplete(document, position, htmlDocument, tagProviders, emmetConfig, fileDocument, autoImportCompletions) {
    const modifierProvider = modifierProvider_1.getModifierProvider();
    const result = {
        isIncomplete: false,
        items: []
    };
    const offset = document.offsetAt(position);
    const node = htmlDocument.findNodeBefore(offset);
    if (!node || (node.isInterpolation && offset <= node.end)) {
        return result;
    }
    const text = document.getText();
    const scanner = htmlScanner_1.createScanner(text, node.start);
    let currentTag;
    let currentAttributeName = '';
    function getReplaceRange(replaceStart, replaceEnd = offset) {
        if (replaceStart > offset) {
            replaceStart = offset;
        }
        return { start: document.positionAt(replaceStart), end: document.positionAt(replaceEnd) };
    }
    function collectOpenTagSuggestions(afterOpenBracket, tagNameEnd) {
        const range = getReplaceRange(afterOpenBracket, tagNameEnd);
        tagProviders.forEach(provider => {
            const priority = provider.priority;
            provider.collectTags((tag, label) => {
                result.items.push({
                    label: tag,
                    kind: vscode_languageserver_types_1.CompletionItemKind.Property,
                    documentation: strings_1.toMarkupContent(label),
                    textEdit: vscode_languageserver_types_1.TextEdit.replace(range, tag),
                    sortText: priority + tag,
                    insertTextFormat: vscode_languageserver_types_1.InsertTextFormat.PlainText
                });
            });
        });
        autoImportCompletions === null || autoImportCompletions === void 0 ? void 0 : autoImportCompletions.forEach(item => {
            result.items.push({
                ...item,
                kind: vscode_languageserver_types_1.CompletionItemKind.Property,
                textEdit: vscode_languageserver_types_1.TextEdit.replace(range, item.label),
                sortText: common_1.Priority.UserCode + item.label,
                insertTextFormat: vscode_languageserver_types_1.InsertTextFormat.PlainText
            });
        });
        return result;
    }
    function getLineIndent(offset) {
        let start = offset;
        while (start > 0) {
            const ch = text.charAt(start - 1);
            if ('\n\r'.indexOf(ch) >= 0) {
                return text.substring(start, offset);
            }
            if (!isWhiteSpace(ch)) {
                return null;
            }
            start--;
        }
        return text.substring(0, offset);
    }
    function collectCloseTagSuggestions(afterOpenBracket, matchingOnly, tagNameEnd = offset) {
        const range = getReplaceRange(afterOpenBracket, tagNameEnd);
        const closeTag = isFollowedBy(text, tagNameEnd, htmlScanner_1.ScannerState.WithinEndTag, htmlScanner_1.TokenType.EndTagClose) ? '' : '>';
        let curr = node;
        while (curr) {
            const tag = curr.tag;
            if (tag && (!curr.closed || (curr.endTagStart && curr.endTagStart > offset))) {
                const item = {
                    label: '/' + tag,
                    kind: vscode_languageserver_types_1.CompletionItemKind.Property,
                    filterText: '/' + tag + closeTag,
                    textEdit: vscode_languageserver_types_1.TextEdit.replace(range, '/' + tag + closeTag),
                    insertTextFormat: vscode_languageserver_types_1.InsertTextFormat.PlainText
                };
                const startIndent = getLineIndent(curr.start);
                const endIndent = getLineIndent(afterOpenBracket - 1);
                if (startIndent !== null && endIndent !== null && startIndent !== endIndent) {
                    const insertText = startIndent + '</' + tag + closeTag;
                    (item.textEdit = vscode_languageserver_types_1.TextEdit.replace(getReplaceRange(afterOpenBracket - 1 - endIndent.length), insertText)),
                        (item.filterText = endIndent + '</' + tag + closeTag);
                }
                result.items.push(item);
                return result;
            }
            curr = curr.parent;
        }
        if (matchingOnly) {
            return result;
        }
        tagProviders.forEach(provider => {
            provider.collectTags((tag, label) => {
                result.items.push({
                    label: '/' + tag,
                    kind: vscode_languageserver_types_1.CompletionItemKind.Property,
                    documentation: strings_1.toMarkupContent(label),
                    filterText: '/' + tag + closeTag,
                    textEdit: vscode_languageserver_types_1.TextEdit.replace(range, '/' + tag + closeTag),
                    insertTextFormat: vscode_languageserver_types_1.InsertTextFormat.PlainText
                });
            });
        });
        return result;
    }
    function collectTagSuggestions(tagStart, tagEnd) {
        collectOpenTagSuggestions(tagStart, tagEnd);
        collectCloseTagSuggestions(tagStart, true, tagEnd);
        return result;
    }
    function collectAttributeNameSuggestions(nameStart, nameEnd = offset) {
        const execArray = /^[:@]/.exec(scanner.getTokenText());
        const filterPrefix = execArray ? execArray[0] : '';
        const start = filterPrefix ? nameStart + 1 : nameStart;
        const range = getReplaceRange(start, nameEnd);
        const value = isFollowedBy(text, nameEnd, htmlScanner_1.ScannerState.AfterAttributeName, htmlScanner_1.TokenType.DelimiterAssign)
            ? ''
            : '="$1"';
        tagProviders.forEach(provider => {
            const priority = provider.priority;
            provider.collectAttributes(currentTag, (attribute, type, documentation) => {
                if ((type === 'event' && filterPrefix !== '@') || (type !== 'event' && filterPrefix === '@')) {
                    return;
                }
                let codeSnippet = attribute;
                if (type !== 'v' && value.length) {
                    codeSnippet = codeSnippet + value;
                }
                if ((filterPrefix === ':' && codeSnippet[0] === ':') || (filterPrefix === '@' && codeSnippet[0] === '@')) {
                    codeSnippet = codeSnippet.slice(1);
                }
                const trimedName = attribute.replace(/^(?::|@)/, '');
                result.items.push({
                    label: attribute,
                    kind: type === 'event' ? vscode_languageserver_types_1.CompletionItemKind.Function : vscode_languageserver_types_1.CompletionItemKind.Value,
                    textEdit: vscode_languageserver_types_1.TextEdit.replace(range, codeSnippet),
                    insertTextFormat: vscode_languageserver_types_1.InsertTextFormat.Snippet,
                    sortText: priority + trimedName,
                    filterText: trimedName,
                    documentation: strings_1.toMarkupContent(documentation)
                });
            });
        });
        const attributeName = scanner.getTokenText();
        if (/\.$/.test(attributeName)) {
            function addModifier(modifiers) {
                modifiers.items.forEach(modifier => {
                    result.items.push({
                        label: modifier.label,
                        kind: vscode_languageserver_types_1.CompletionItemKind.Method,
                        textEdit: vscode_languageserver_types_1.TextEdit.insert(document.positionAt(nameEnd), modifier.label),
                        insertTextFormat: vscode_languageserver_types_1.InsertTextFormat.Snippet,
                        sortText: modifiers.priority + modifier.label,
                        documentation: strings_1.toMarkupContent(modifier.documentation)
                    });
                });
            }
            if (attributeName.startsWith('@') || attributeName.startsWith('v-on')) {
                addModifier(modifierProvider.eventModifiers);
            }
            const execArray = /^(?:@|v-on:)([A-Za-z]*)\.?/.exec(attributeName);
            const eventName = execArray && execArray[1] ? execArray[1] : '';
            const keyEvent = ['keydown', 'keypress', 'keyup'];
            if (keyEvent.includes(eventName)) {
                addModifier(modifierProvider.keyModifiers);
                addModifier(modifierProvider.systemModifiers);
            }
            const mouseEvent = ['click', 'dblclick', 'mouseup', 'mousedown'];
            if (mouseEvent.includes(eventName)) {
                addModifier(modifierProvider.mouseModifiers);
                addModifier(modifierProvider.systemModifiers);
            }
            if (attributeName.startsWith('v-bind') || attributeName.startsWith(':')) {
                addModifier(modifierProvider.propsModifiers);
            }
            if (attributeName.startsWith('v-model')) {
                addModifier(modifierProvider.vModelModifiers);
            }
        }
        return result;
    }
    function collectAttributeValueSuggestions(attr, valueStart, valueEnd) {
        if (attr.startsWith('v-') || attr.startsWith('@') || attr.startsWith(':')) {
            return nullMode_1.NULL_COMPLETION;
        }
        let range;
        let addQuotes;
        if (valueEnd && offset > valueStart && offset <= valueEnd && text[valueStart] === '"') {
            // inside attribute
            if (valueEnd > offset && text[valueEnd - 1] === '"') {
                valueEnd--;
            }
            const wsBefore = getWordStart(text, offset, valueStart + 1);
            const wsAfter = getWordEnd(text, offset, valueEnd);
            range = getReplaceRange(wsBefore, wsAfter);
            addQuotes = false;
        }
        else {
            range = getReplaceRange(valueStart, valueEnd);
            addQuotes = true;
        }
        const attribute = currentAttributeName.toLowerCase();
        tagProviders.forEach(provider => {
            provider.collectValues(currentTag, attribute, value => {
                const insertText = addQuotes ? '"' + value + '"' : value;
                result.items.push({
                    label: value,
                    filterText: insertText,
                    kind: vscode_languageserver_types_1.CompletionItemKind.Unit,
                    textEdit: vscode_languageserver_types_1.TextEdit.replace(range, insertText),
                    insertTextFormat: vscode_languageserver_types_1.InsertTextFormat.PlainText
                });
            });
        });
        return result;
    }
    function scanNextForEndPos(nextToken) {
        if (offset === scanner.getTokenEnd()) {
            token = scanner.scan();
            if (token === nextToken && scanner.getTokenOffset() === offset) {
                return scanner.getTokenEnd();
            }
        }
        return offset;
    }
    let token = scanner.scan();
    while (token !== htmlScanner_1.TokenType.EOS && scanner.getTokenOffset() <= offset) {
        switch (token) {
            case htmlScanner_1.TokenType.StartTagOpen:
                if (scanner.getTokenEnd() === offset) {
                    const endPos = scanNextForEndPos(htmlScanner_1.TokenType.StartTag);
                    const range = getReplaceRange(offset, endPos);
                    const list = collectTagSuggestions(offset, endPos);
                    addComponent(list, fileDocument, range);
                    return list;
                }
                break;
            case htmlScanner_1.TokenType.StartTag:
                if (scanner.getTokenOffset() <= offset && offset <= scanner.getTokenEnd()) {
                    return collectOpenTagSuggestions(scanner.getTokenOffset(), scanner.getTokenEnd());
                }
                currentTag = scanner.getTokenText();
                break;
            case htmlScanner_1.TokenType.AttributeName:
                if (scanner.getTokenOffset() <= offset && offset <= scanner.getTokenEnd()) {
                    return collectAttributeNameSuggestions(scanner.getTokenOffset(), scanner.getTokenEnd());
                }
                currentAttributeName = scanner.getTokenText();
                break;
            case htmlScanner_1.TokenType.DelimiterAssign:
                if (scanner.getTokenEnd() === offset) {
                    return collectAttributeValueSuggestions(currentAttributeName, scanner.getTokenEnd());
                }
                break;
            case htmlScanner_1.TokenType.AttributeValue:
                if (scanner.getTokenOffset() <= offset && offset <= scanner.getTokenEnd()) {
                    if (currentAttributeName === 'style') {
                        const emmetCompletions = emmet.doComplete(document, position, 'css', emmetConfig);
                        return emmetCompletions || nullMode_1.NULL_COMPLETION;
                    }
                    else {
                        return collectAttributeValueSuggestions(currentAttributeName, scanner.getTokenOffset(), scanner.getTokenEnd());
                    }
                }
                break;
            case htmlScanner_1.TokenType.Whitespace:
                if (offset <= scanner.getTokenEnd()) {
                    switch (scanner.getScannerState()) {
                        case htmlScanner_1.ScannerState.AfterOpeningStartTag:
                            const startPos = scanner.getTokenOffset();
                            const endTagPos = scanNextForEndPos(htmlScanner_1.TokenType.StartTag);
                            return collectTagSuggestions(startPos, endTagPos);
                        case htmlScanner_1.ScannerState.WithinTag:
                        case htmlScanner_1.ScannerState.AfterAttributeName:
                            return collectAttributeNameSuggestions(scanner.getTokenEnd());
                        case htmlScanner_1.ScannerState.BeforeAttributeValue:
                            return collectAttributeValueSuggestions(currentAttributeName, scanner.getTokenEnd());
                        case htmlScanner_1.ScannerState.AfterOpeningEndTag:
                            return collectCloseTagSuggestions(scanner.getTokenOffset() - 1, false);
                    }
                }
                break;
            case htmlScanner_1.TokenType.EndTagOpen:
                if (offset <= scanner.getTokenEnd()) {
                    const afterOpenBracket = scanner.getTokenOffset() + 1;
                    const endOffset = scanNextForEndPos(htmlScanner_1.TokenType.EndTag);
                    return collectCloseTagSuggestions(afterOpenBracket, false, endOffset);
                }
                break;
            case htmlScanner_1.TokenType.EndTag:
                if (offset <= scanner.getTokenEnd()) {
                    let start = scanner.getTokenOffset() - 1;
                    while (start >= 0) {
                        const ch = text.charAt(start);
                        if (ch === '/') {
                            return collectCloseTagSuggestions(start, false, scanner.getTokenEnd());
                        }
                        else if (!isWhiteSpace(ch)) {
                            break;
                        }
                        start--;
                    }
                }
                break;
            case htmlScanner_1.TokenType.Content:
                if (offset <= scanner.getTokenEnd()) {
                    const completionList = emmet.doComplete(document, position, 'html', emmetConfig);
                    if (completionList) {
                        const items = completionList.items;
                        if (items && items.length > 0) {
                            items.forEach(item => {
                                if (item.label === 'image') {
                                    item.documentation = `<image src=""/>`;
                                    if (item.textEdit) {
                                        item.textEdit.newText = `<image src="" />`;
                                    }
                                }
                            });
                        }
                    }
                    const list = overrideCompletionList(completionList, fileDocument, position);
                    return list;
                }
                break;
            default:
                if (offset <= scanner.getTokenEnd()) {
                    return result;
                }
                break;
        }
        token = scanner.scan();
    }
    return result;
}
exports.doComplete = doComplete;
function addComponent(list, document, range) {
    const uri = vscode_uri_1.URI.parse(document.uri);
    const projectPath = workspace_1.getProjectPath(uri.fsPath);
    if (projectPath) {
        const items = list.items;
        const labelJson = fs.readFileSync(path.join(projectPath, '.yonstudio', '.labelinfo'), { encoding: 'utf-8' });
        const label = JSON.parse(labelJson);
        const text = document.getText();
        const index = text.indexOf('<script>');
        for (const key in label) {
            const cpath = label[key];
            if (cpath !== uri.fsPath) {
                let relative = os.platform() === 'win32' ? path.relative(path.dirname(uri.fsPath), label[key]).replace(/\\/g, '/') : path.relative(path.dirname(uri.fsPath), label[key]);
                if (!relative.includes('./')) {
                    relative = './' + relative;
                }
                // 判断是否已经导入组件
                let detail = '组件';
                let additionalTextEdits = [];
                if (!text.includes(relative)) {
                    detail = '自动导入组件';
                    const additionalText = `\nimport "${relative}"\n`;
                    const textEdit = vscode_languageserver_types_1.TextEdit.insert(document.positionAt(index + 1 + '<script>'.length), additionalText);
                    additionalTextEdits.push(textEdit);
                }
                const item = {
                    label: key,
                    detail,
                    documentation: `<${key}></${key}>`,
                    insertTextFormat: vscode_languageserver_types_1.InsertTextFormat.Snippet,
                    textEdit: vscode_languageserver_types_1.TextEdit.replace(range, `${key}>\${0}</${key}>`),
                    additionalTextEdits
                };
                items.push(item);
            }
        }
    }
}
function overrideCompletionList(completionList, document, position) {
    const projectPath = workspace_1.getProjectPath(vscode_uri_1.URI.parse(document.uri).fsPath);
    const result = {
        isIncomplete: true,
        items: []
    };
    if (fs.existsSync(path.join(projectPath, '.yonstudio', '.labelinfo'))) {
        const currentLineTillPosition = getCurrentLine(document, position).substr(0, position.character);
        const currentWord = getCurrentWord(currentLineTillPosition);
        const uri = vscode_uri_1.URI.parse(document.uri).fsPath;
        if (currentWord) {
            const labelJson = fs.readFileSync(path.join(projectPath, '.yonstudio', '.labelinfo'), { encoding: 'utf-8' });
            const label = JSON.parse(labelJson);
            const range = vscode_languageserver_types_1.Range.create({ line: position.line, character: position.character - currentWord.length }, position);
            const text = document.getText();
            const index = text.indexOf('<script>');
            for (const key in label) {
                const cPath = label[key];
                // 检测是否是当前的组件
                if (key.includes(currentWord) && !uri.includes(cPath)) {
                    let relative = os.platform() === 'win32' ? path.relative(path.dirname(uri), label[key]).replace(/\\/g, '/') : path.relative(path.dirname(uri), label[key]);
                    if (!relative.includes('./')) {
                        relative = './' + relative;
                    }
                    // 判断是否已经导入组件
                    let detail = '组件';
                    let additionalTextEdits = [];
                    if (!text.includes(relative)) {
                        detail = '自动导入组件';
                        const additionalText = `\nimport "${relative}"\n`;
                        const textEdit = vscode_languageserver_types_1.TextEdit.insert(document.positionAt(index + 1 + '<script>'.length), additionalText);
                        additionalTextEdits.push(textEdit);
                    }
                    const item = {
                        label: key,
                        detail,
                        documentation: `<${key}></${key}>`,
                        textEdit: vscode_languageserver_types_1.TextEdit.replace(range, `<${key}>\${0}</${key}>`),
                        insertTextFormat: vscode_languageserver_types_1.InsertTextFormat.Snippet,
                        additionalTextEdits
                    };
                    if (completionList) {
                        completionList.items.push(item);
                    }
                    else {
                        result.items.push(item);
                    }
                }
            }
        }
    }
    return completionList ? completionList : result;
}
function isWhiteSpace(s) {
    return /^\s*$/.test(s);
}
function isFollowedBy(s, offset, intialState, expectedToken) {
    const scanner = htmlScanner_1.createScanner(s, offset, intialState);
    let token = scanner.scan();
    while (token === htmlScanner_1.TokenType.Whitespace) {
        token = scanner.scan();
    }
    return token === expectedToken;
}
function getWordStart(s, offset, limit) {
    while (offset > limit && !isWhiteSpace(s[offset - 1])) {
        offset--;
    }
    return offset;
}
function getWordEnd(s, offset, limit) {
    while (offset < limit && !isWhiteSpace(s[offset])) {
        offset++;
    }
    return offset;
}
function getCurrentLine(document, position) {
    const offset = document.offsetAt(position);
    const text = document.getText();
    let start = 0;
    let end = text.length;
    for (let i = offset - 1; i >= 0; i--) {
        if (text[i] === '\n') {
            start = i + 1;
            break;
        }
    }
    for (let i = offset; i < text.length; i++) {
        if (text[i] === '\n') {
            end = i;
            break;
        }
    }
    return text.substring(start, end);
}
function getCurrentWord(currentLineTillPosition) {
    if (currentLineTillPosition) {
        const matches = currentLineTillPosition.match(/[\w,:,-,\.]*$/);
        if (matches) {
            return matches[0];
        }
    }
    return undefined;
}
//# sourceMappingURL=htmlCompletion.js.map