"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAutoImportSfcPlugin = void 0;
const lodash_1 = require("lodash");
const os_1 = require("os");
const path_1 = require("path");
const vscode_languageserver_textdocument_1 = require("vscode-languageserver-textdocument");
const vscode_languageserver_types_1 = require("vscode-languageserver-types");
const strings_1 = require("../../utils/strings");
/**
 * Auto import component in script when completion in template.
 * ## Implementation:
 * 1. Get all vue files path from ts language service host.
 * 2. Record more data in VueInfoService. Like: componentsDefine position.
 * 3. Provide completion list in template from vue files path
 * 4. Mock code to trigger auto import in TS for make importing component TextEdit.
 * 5. Add component define TextEdit from second step.
 * 6. Provide completion/resolve from fourth and fifth steps.
 *
 * ## Example
 * For mock code example in TS when componentName is `Item`.
 * The `=` is position for call completion/resolve in TS language service.
 * ```typescript
 * export default {
 *   components: {
 *     Item: ItemV=
 *   }
 * }
 * ```
 */
function createAutoImportSfcPlugin(tsModule, vueInfoService) {
    let getConfigure;
    let getVueFiles;
    let getJSResolve;
    let getTSScriptTarget = () => undefined;
    function createMockDoc(document, componentInsertPos, mockPartContent) {
        const mockDocContent = document.getText().slice(0, componentInsertPos) + mockPartContent + document.getText().slice(componentInsertPos);
        const mockDoc = vscode_languageserver_textdocument_1.TextDocument.create(document.uri, document.languageId, document.version + 1, mockDocContent);
        return mockDoc;
    }
    function getJSImportEdits(nameForTriggerResolveInTs, mockDoc, offset, item, componentName) {
        var _a, _b;
        const mockCompletionItem = {
            label: nameForTriggerResolveInTs + 'Vue',
            data: {
                languageId: 'vue-html',
                uri: mockDoc.uri,
                offset,
                source: item.data.path
            }
        };
        const textEdits = (_b = (_a = getJSResolve(mockDoc, mockCompletionItem)) === null || _a === void 0 ? void 0 : _a.additionalTextEdits) === null || _b === void 0 ? void 0 : _b.map(edit => {
            edit.newText = edit.newText.replace(nameForTriggerResolveInTs + 'Vue', componentName);
            return edit;
        });
        return textEdits;
    }
    function getNoDuplicateComponentName(childComponents, componentCompletionName) {
        let index = 1;
        while (childComponents.some(el => el.name.toLowerCase() === componentCompletionName.toLowerCase())) {
            componentCompletionName = `${componentCompletionName}${index++}`;
        }
        return lodash_1.upperFirst(componentCompletionName);
    }
    return {
        setGetConfigure(fn) {
            getConfigure = fn;
        },
        setGetFilesFn(fn) {
            getVueFiles = fn;
        },
        setGetJSResolve(fn) {
            getJSResolve = fn;
        },
        setGetTSScriptTarget(fn) {
            getTSScriptTarget = fn;
        },
        doComplete(document) {
            var _a, _b, _c;
            const config = getConfigure();
            if (!config.stml.completion.autoImport) {
                return [];
            }
            if (!getVueFiles || !getJSResolve || !vueInfoService) {
                return [];
            }
            const childComponentsPath = (_c = (_b = (_a = vueInfoService.getInfo(document)) === null || _a === void 0 ? void 0 : _a.componentInfo.childComponents) === null || _b === void 0 ? void 0 : _b.map(el => { var _a; return (_a = el.definition) === null || _a === void 0 ? void 0 : _a.path; })) !== null && _c !== void 0 ? _c : [];
            return getVueFiles()
                .filter(fileName => !childComponentsPath.includes(fileName))
                .map(fileName => {
                let tagName = path_1.basename(fileName, '.stml');
                if (config.stml.completion.tagCasing === 'kebab') {
                    tagName = lodash_1.kebabCase(tagName);
                }
                const documentation = `
\`\`\`typescript
import ${lodash_1.upperFirst(lodash_1.camelCase(tagName))} from '${fileName}'
\`\`\`
`;
                return {
                    label: tagName,
                    insertText: tagName,
                    documentation: strings_1.toMarkupContent(documentation),
                    data: {
                        languageId: 'vue-html',
                        uri: document.uri,
                        isFromAutoImportVueService: true,
                        path: fileName
                    }
                };
            });
        },
        isMyResolve(item) {
            var _a, _b;
            return (_b = (_a = item.data) === null || _a === void 0 ? void 0 : _a.isFromAutoImportVueService) !== null && _b !== void 0 ? _b : false;
        },
        doResolve(document, item) {
            var _a, _b, _c;
            const config = getConfigure();
            const componentInfo = (_a = vueInfoService === null || vueInfoService === void 0 ? void 0 : vueInfoService.getInfo(document)) === null || _a === void 0 ? void 0 : _a.componentInfo;
            if (!componentInfo) {
                return item;
            }
            const componentDefine = componentInfo === null || componentInfo === void 0 ? void 0 : componentInfo.componentsDefine;
            const childComponents = (_b = componentInfo === null || componentInfo === void 0 ? void 0 : componentInfo.childComponents) === null || _b === void 0 ? void 0 : _b.filter(c => !c.global);
            const nameForTriggerResolveInTs = strings_1.modulePathToValidIdentifier(tsModule, item.data.path, (_c = getTSScriptTarget()) !== null && _c !== void 0 ? _c : tsModule.ScriptTarget.ESNext);
            /**
             * have `components` property case
             */
            if (componentDefine && childComponents) {
                const componentName = getNoDuplicateComponentName(childComponents, nameForTriggerResolveInTs);
                const componentInsertPos = componentDefine === null || componentDefine === void 0 ? void 0 : componentDefine.insertPos;
                const mockPartContent = `${nameForTriggerResolveInTs}: ${nameForTriggerResolveInTs.slice(0, nameForTriggerResolveInTs.length - 2)}`;
                const mockDoc = createMockDoc(document, componentInsertPos, mockPartContent);
                const textEdits = getJSImportEdits(nameForTriggerResolveInTs, mockDoc, componentInsertPos + mockPartContent.length, item, componentName);
                if (textEdits) {
                    const currentComponentText = document.getText().slice(componentDefine.start, componentDefine.end);
                    const newInsertTexts = [];
                    if (document.getText().charAt(componentInsertPos - 1) !== ',' && childComponents.length !== 0) {
                        newInsertTexts.push(',');
                    }
                    if (currentComponentText.includes('\n')) {
                        newInsertTexts.push(os_1.EOL);
                        const textInLine = document
                            .getText()
                            .slice(document.getText().lastIndexOf('\n', componentDefine.start), document.getText().indexOf('\n', componentDefine.start));
                        newInsertTexts.push(textInLine.slice(1, textInLine.slice(1).search(/[^ ]/) + 1).repeat(2));
                    }
                    else if (childComponents.length !== 0) {
                        newInsertTexts.push(' ');
                    }
                    newInsertTexts.push(componentName);
                    textEdits.push(vscode_languageserver_types_1.TextEdit.insert(document.positionAt(componentInsertPos), newInsertTexts.join('')));
                }
                item.additionalTextEdits = textEdits;
                item.insertText = config.stml.completion.tagCasing === 'initial' ? componentName : lodash_1.kebabCase(componentName);
            }
            else if (componentInfo.insertInOptionAPIPos) {
                /**
                 * no have `components` property case
                 */
                const componentName = nameForTriggerResolveInTs;
                const mockEndPart = '},';
                const mockPartContent = `components: {${nameForTriggerResolveInTs}: ` +
                    nameForTriggerResolveInTs.slice(0, nameForTriggerResolveInTs.length - 2) +
                    mockEndPart;
                const mockDoc = createMockDoc(document, componentInfo.insertInOptionAPIPos, mockPartContent);
                const textEdits = getJSImportEdits(nameForTriggerResolveInTs, mockDoc, componentInfo.insertInOptionAPIPos + mockPartContent.length - mockEndPart.length, item, componentName);
                if (textEdits) {
                    const newInsertTexts = [os_1.EOL];
                    if (config.stml.format.options.useTabs) {
                        newInsertTexts.push('\t');
                    }
                    else {
                        newInsertTexts.push(' '.repeat(config.stml.format.options.tabSize));
                    }
                    newInsertTexts.push(`components: { ${componentName} },`);
                    textEdits.push(vscode_languageserver_types_1.TextEdit.insert(document.positionAt(componentInfo.insertInOptionAPIPos), newInsertTexts.join('')));
                }
                item.additionalTextEdits = textEdits;
                item.insertText = config.stml.completion.tagCasing === 'initial' ? componentName : lodash_1.kebabCase(componentName);
            }
            return item;
        }
    };
}
exports.createAutoImportSfcPlugin = createAutoImportSfcPlugin;
//# sourceMappingURL=autoImportSfcPlugin.js.map