"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LanguageModes = void 0;
const languageModelCache_1 = require("./languageModelCache");
const embeddedSupport_1 = require("./embeddedSupport");
const vue_1 = require("../modes/vue");
const style_1 = require("../modes/style");
const javascript_1 = require("../modes/script/javascript");
const template_1 = require("../modes/template");
const stylus_1 = require("../modes/style/stylus");
const nullMode_1 = require("../modes/nullMode");
const serviceHost_1 = require("../services/typescriptService/serviceHost");
const sassLanguageMode_1 = require("../modes/style/sass/sassLanguageMode");
const pug_1 = require("../modes/pug");
const autoImportSfcPlugin_1 = require("../modes/plugins/autoImportSfcPlugin");
class LanguageModes {
    constructor() {
        this.modes = {
            vue: nullMode_1.nullMode,
            pug: nullMode_1.nullMode,
            'vue-html': nullMode_1.nullMode,
            css: nullMode_1.nullMode,
            postcss: nullMode_1.nullMode,
            scss: nullMode_1.nullMode,
            less: nullMode_1.nullMode,
            sass: nullMode_1.nullMode,
            stylus: nullMode_1.nullMode,
            javascript: nullMode_1.nullMode,
            typescript: nullMode_1.nullMode,
            tsx: nullMode_1.nullMode
        };
        this.documentRegions = languageModelCache_1.getLanguageModelCache(10, 60, document => embeddedSupport_1.getVueDocumentRegions(document));
        this.modelCaches = [];
        this.modelCaches.push(this.documentRegions);
    }
    getFilesFn() {
        let strs = [];
        if (this.serviceHost) {
            strs = this.serviceHost.getFileNames().filter(fileName => fileName.endsWith('.stml'));
        }
        return strs;
    }
    getTsScriptTarget() {
        let target = undefined;
        if (this.serviceHost) {
            target = this.serviceHost.getComplierOptions().target;
        }
        return target;
    }
    async init(env, services, globalSnippetDir) {
        const tsModule = services.dependencyService.get('typescript').module;
        /**
         * Documents where everything outside `<script>` is replaced with whitespace
         */
        const scriptRegionDocuments = languageModelCache_1.getLanguageModelCache(10, 60, document => {
            const vueDocument = this.documentRegions.refreshAndGet(document);
            return vueDocument.getSingleTypeDocument('script');
        });
        this.serviceHost = serviceHost_1.getServiceHost(tsModule, env, scriptRegionDocuments);
        const autoImportSfcPlugin = autoImportSfcPlugin_1.createAutoImportSfcPlugin(tsModule, services.infoService);
        autoImportSfcPlugin.setGetTSScriptTarget(() => this.serviceHost.getComplierOptions().target);
        autoImportSfcPlugin.setGetFilesFn(() => this.serviceHost.getFileNames().filter(fileName => fileName.endsWith('.stml')));
        const vueHtmlMode = new template_1.VueHTMLMode(tsModule, this.serviceHost, env, this.documentRegions, autoImportSfcPlugin, services.dependencyService, services.infoService);
        const jsMode = await javascript_1.getJavascriptMode(this.serviceHost, env, this.documentRegions, services.dependencyService, env.getGlobalComponentInfos(), services.infoService);
        autoImportSfcPlugin.setGetConfigure(env.getConfig);
        autoImportSfcPlugin.setGetJSResolve(jsMode.doResolve);
        this.modes['vue'] = vue_1.getVueMode(env, globalSnippetDir);
        this.modes['vue-html'] = vueHtmlMode;
        this.modes['pug'] = pug_1.getPugMode(env, services.dependencyService);
        this.modes['css'] = style_1.getCSSMode(env, this.documentRegions, services.dependencyService);
        this.modes['postcss'] = style_1.getPostCSSMode(env, this.documentRegions, services.dependencyService);
        this.modes['scss'] = style_1.getSCSSMode(env, this.documentRegions, services.dependencyService);
        this.modes['sass'] = new sassLanguageMode_1.SassLanguageMode(env);
        this.modes['less'] = style_1.getLESSMode(env, this.documentRegions, services.dependencyService);
        this.modes['stylus'] = stylus_1.getStylusMode(env, this.documentRegions, services.dependencyService);
        this.modes['javascript'] = jsMode;
        this.modes['typescript'] = jsMode;
        this.modes['tsx'] = jsMode;
    }
    getModeAtPosition(document, position) {
        const languageId = this.documentRegions.refreshAndGet(document).getLanguageAtPosition(position);
        return this.modes[languageId];
    }
    getAllLanguageModeRangesInDocument(document) {
        const result = [];
        const documentRegions = this.documentRegions.refreshAndGet(document);
        documentRegions.getAllLanguageRanges().forEach(lr => {
            const mode = this.modes[lr.languageId];
            if (mode) {
                result.push({
                    mode,
                    ...lr
                });
            }
        });
        return result;
    }
    getAllModes() {
        const result = [];
        for (const languageId in this.modes) {
            const mode = this.modes[languageId];
            if (mode) {
                result.push(mode);
            }
        }
        return result;
    }
    getMode(languageId) {
        return this.modes[languageId];
    }
    onDocumentRemoved(document) {
        this.modelCaches.forEach(mc => mc.onDocumentRemoved(document));
        for (const mode in this.modes) {
            this.modes[mode].onDocumentRemoved(document);
        }
    }
    dispose() {
        this.modelCaches.forEach(mc => mc.dispose());
        this.modelCaches = [];
        for (const mode in this.modes) {
            this.modes[mode].dispose();
        }
        if (this.serviceHost) {
            this.serviceHost.dispose();
        }
    }
}
exports.LanguageModes = LanguageModes;
//# sourceMappingURL=languageModes.js.map