/**
 * Expands given abbreviation into string, formatted according to provided
 * syntax and options
 * @param  {String|Node} abbr        Abbreviation string or parsed abbreviation tree
 * @param  {String|Object} [options] Parsing and formatting options (object) or
 * abbreviation syntax (string)
 * @return {String}
 */
export function expand(abbr: string | Node, options?: string | any): string;
/**
 * Parses given abbreviation into AST tree. This tree can be later formatted to
 * string with `expand` function
 * @param  {String} abbr             Abbreviation to parse
 * @param  {String|Object} [options] Parsing and formatting options (object) or
 * abbreviation syntax (string)
 * @return {Node}
 */
export function parse(abbr: string, options?: string | any): Node;
/**
 * Creates snippets registry for given syntax and additional `snippets`
 * @param  {String} syntax   Snippets syntax, used for retreiving predefined snippets
 * @param  {SnippetsRegistry|Object|Object[]} [snippets] Additional snippets
 * @return {SnippetsRegistry}
 */
export function createSnippetsRegistry(syntax: string, snippets?: SnippetsRegistry | any | any[]): SnippetsRegistry;
export function createOptions(options: any): any;
/**
 * Check if given syntax belongs to stylesheet markup.
 * Emmet uses different abbreviation flavours: one is a default markup syntax,
 * used for HTML, Slim, Pug etc, the other one is used for stylesheets and
 * allows embedded values in abbreviation name
 * @param  {String}  syntax
 * @return {Boolean}
 */
export function isStylesheet(syntax: string): boolean;
/**
 * Creates output profile from given options
 * @param  {Object} options
 * @return {Profile}
 */
export function createProfile(options: any): Profile;
/**
 * A parsed abbreviation AST node. Nodes build up an abbreviation AST tree
 */
export class Node {
    /**
     * Creates a new node
     * @param {String} [name] Node name
     * @param {Array} [attributes] Array of attributes to add
     */
    constructor(name?: string, attributes?: any[]);
    name: string;
    value: any;
    repeat: any;
    selfClosing: boolean;
    children: any[];
    /** @type {Node} Pointer to parent node */
    parent: Node;
    /** @type {Node} Pointer to next sibling */
    next: Node;
    /** @type {Node} Pointer to previous sibling */
    previous: Node;
    _attributes: any[];
    /**
     * Array of current node attributes
     * @return {Attribute[]} Array of attributes
     */
    get attributes(): Attribute[];
    /**
     * A shorthand to retreive node attributes as map
     * @return {Object}
     */
    get attributesMap(): any;
    /**
     * Check if current node is a grouping one, e.g. has no actual representation
     * and is used for grouping subsequent nodes only
     * @return {Boolean}
     */
    get isGroup(): boolean;
    /**
     * Check if given node is a text-only node, e.g. contains only value
     * @return {Boolean}
     */
    get isTextOnly(): boolean;
    /**
     * Returns first child node
     * @return {Node}
     */
    get firstChild(): Node;
    /**
     * Returns last child of current node
     * @return {Node}
     */
    get lastChild(): Node;
    /**
     * Return index of current node in its parent child list
     * @return {Number} Returns -1 if current node is a root one
     */
    get childIndex(): number;
    /**
     * Returns next sibling of current node
     * @return {Node}
     */
    get nextSibling(): Node;
    /**
     * Returns previous sibling of current node
     * @return {Node}
     */
    get previousSibling(): Node;
    /**
     * Returns array of unique class names in current node
     * @return {String[]}
     */
    get classList(): string[];
    /**
     * Convenient alias to create a new node instance
     * @param {String} [name] Node name
     * @param {Object} [attributes] Attributes hash
     * @return {Node}
     */
    create(name?: string, attributes?: any): Node;
    /**
     * Sets given attribute for current node
     * @param {String|Object|Attribute} name Attribute name or attribute object
     * @param {String} [value] Attribute value
     */
    setAttribute(name: string | any | Attribute, value?: string): void;
    /**
     * Check if attribute with given name exists in node
     * @param  {String} name
     * @return {Boolean}
     */
    hasAttribute(name: string): boolean;
    /**
     * Returns attribute object by given name
     * @param  {String} name
     * @return {Attribute}
     */
    getAttribute(name: string): Attribute;
    /**
     * Replaces attribute with new instance
     * @param {String|Attribute} curAttribute Current attribute name or instance
     * to replace
     * @param {String|Object|Attribute} newName New attribute name or attribute object
     * @param {String} [newValue] New attribute value
     */
    replaceAttribute(curAttribute: string | Attribute, newName: string | any | Attribute, newValue?: string): void;
    /**
     * Removes attribute with given name
     * @param  {String|Attribute} attr Atrtibute name or instance
     */
    removeAttribute(attr: string | Attribute): void;
    /**
     * Removes all attributes from current node
     */
    clearAttributes(): void;
    /**
     * Adds given class name to class attribute
     * @param {String} token Class name token
     */
    addClass(token: string): void;
    /**
     * Check if current node contains given class name
     * @param {String} token Class name token
     * @return {Boolean}
     */
    hasClass(token: string): boolean;
    /**
     * Removes given class name from class attribute
     * @param {String} token Class name token
     */
    removeClass(token: string): void;
    /**
     * Appends child to current node
     * @param {Node} node
     */
    appendChild(node: Node): void;
    /**
     * Inserts given `newNode` before `refNode` child node
     * @param {Node} newNode
     * @param {Node} refNode
     */
    insertBefore(newNode: Node, refNode: Node): void;
    /**
     * Insert given `node` at `pos` position of child list
     * @param {Node} node
     * @param {Number} pos
     */
    insertAt(node: Node, pos: number): void;
    /**
     * Removes given child from current node
     * @param {Node} node
     */
    removeChild(node: Node): void;
    /**
     * Removes current node from its parent
     */
    remove(): void;
    /**
     * Creates a detached copy of current node
     * @param {Boolean} deep Clone node contents as well
     * @return {Node}
     */
    clone(deep: boolean): Node;
    /**
     * Walks on each descendant node and invokes given `fn` function on it.
     * The function receives two arguments: the node itself and its depth level
     * from current node. If function returns `false`, it stops walking
     * @param {Function} fn
     */
    walk(fn: Function, _level: any): boolean;
    /**
     * A helper method for transformation chaining: runs given `fn` function on
     * current node and returns the same node
     */
    use(fn: any, ...args: any[]): Node;
    toString(): string;
}
/**
 * A snippets registry. Contains snippets, separated by store and sorted by
 * priority: a store with higher priority takes precedence when resolving snippet
 * for given key
 */
declare class SnippetsRegistry {
    /**
     * Creates snippets registry, filled with given `data`
     * @param {Object|Array} data Registry snippets. If array is given, adds items
     * from array in order of precedence, registers global snippets otherwise
     */
    constructor(data: any | Array);
    _registry: any[];
    /**
     * Return store for given level
     * @param {Number} level
     * @return {SnippetsStorage}
     */
    get(level: number): SnippetsStorage;
    /**
     * Adds new store for given level
     * @param {Number} [level] Store level (priority). Store with higher level
     * takes precedence when resolving snippets
     * @param {Object} [snippets] A snippets data for new store
     * @return {SnipetsStorage}
     */
    add(level?: number, snippets?: any): any;
    /**
     * Remove registry with given level or store
     * @param {Number|SnippetsStorage} data Either level or snippets store
     */
    remove(data: number | SnippetsStorage): void;
    /**
     * Returns snippet from registry that matches given name
     * @param {String} name
     * @return {Snippet}
     */
    resolve(name: string): Snippet;
    /**
     * Returns all available snippets from current registry. Snippets with the
     * same key are resolved by their storage priority.
     * @param {Object} options
     * @param {Object} options.type Return snippets only of given type: 'string'
     * or 'regexp'. Returns all snippets if not defined
     * @return {Array}
     */
    all(options: {
        type: any;
    }): any[];
    /**
     * Removes all stores from registry
     */
    clear(): void;
}
/**
 * Creates output profile for given options (@see defaults)
 * @param {defaults} options
 */
declare class Profile {
    constructor(options: any);
    options: any;
    quoteChar: string;
    /**
     * Returns value of given option name
     * @param {String} name
     * @return {*}
     */
    get(name: string): any;
    /**
     * Quote given string according to profile
     * @param {String} str String to quote
     * @return {String}
     */
    quote(str: string): string;
    /**
     * Output given tag name accoding to options
     * @param {String} name
     * @return {String}
     */
    name(name: string): string;
    /**
     * Outputs attribute name accoding to current settings
     * @param {String} Attribute name
     * @return {String}
     */
    attribute(attr: any): string;
    /**
     * Check if given attribute is boolean
     * @param {Attribute} attr
     * @return {Boolean}
     */
    isBooleanAttribute(attr: Attribute): boolean;
    /**
     * Returns a token for self-closing tag, depending on current options
     * @return {String}
     */
    selfClose(): string;
    /**
     * Returns indent for given level
     * @param {Number} level Indentation level
     * @return {String}
     */
    indent(level: number): string;
    /**
     * Check if given tag name belongs to inline-level element
     * @param {Node|String} node Parsed node or tag name
     * @return {Boolean}
     */
    isInline(node: Node | string): boolean;
    /**
     * Outputs formatted field for given params
     * @param {Number} index Field index
     * @param {String} [placeholder] Field placeholder, can be empty
     * @return {String}
     */
    field(index: number, placeholder?: string): string;
}
/**
 * Attribute descriptor of parsed abbreviation node
 * @param {String} name Attribute name
 * @param {String} value Attribute value
 * @param {Object} options Additional custom attribute options
 * @param {Boolean} options.boolean Attribute is boolean (e.g. name equals value)
 * @param {Boolean} options.implied Attribute is implied (e.g. must be outputted
 * only if contains non-null value)
 */
declare class Attribute {
    constructor(name: any, value: any, options: any);
    name: any;
    value: any;
    options: any;
    /**
     * Create a copy of current attribute
     * @return {Attribute}
     */
    clone(): Attribute;
    /**
     * A string representation of current node
     */
    valueOf(): string;
}
declare class SnippetsStorage {
    constructor(data: any);
    _string: Map<any, any>;
    _regexp: Map<any, any>;
    _disabled: boolean;
    get disabled(): boolean;
    /**
     * Disables current store. A disabled store always returns `undefined`
     * on `get()` method
     */
    disable(): void;
    /**
     * Enables current store.
     */
    enable(): void;
    /**
     * Registers a new snippet item
     * @param {String|Regexp} key
     * @param {String|Function} value
     */
    set(key: string | any, value: string | Function): SnippetsStorage;
    /**
     * Returns a snippet matching given key. It first tries to find snippet
     * exact match in a string key map, then tries to match one with regexp key
     * @param {String} key
     * @return {Snippet}
     */
    get(key: string): Snippet;
    /**
     * Batch load of snippets data
     * @param {Object|Map} data
     */
    load(data: any | Map): void;
    /**
     * Clears all stored snippets
     */
    reset(): void;
    /**
     * Returns all available snippets from given store
     */
    values(): any[];
}
declare class Snippet {
    constructor(key: any, value: any);
    key: any;
    value: any;
}
export {};
