"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = format;

var _path = require("path");

var _fs = require("fs");

var _utils = require("./utils");

/**
 * Formats the text with prettier and then eslint based on the given options
 * @param {String} options.filePath - the path of the file being formatted
 *  can be used in leu of `eslintConfig` (eslint will be used to find the
 *  relevant config for the file). Will also be used to load the `text` if
 *  `text` is not provided.
 * @param {String} options.text - the text (TypeScript code) to format
 * @param {String} options.tslintPath - the path to the tslint module to use.
 *   Will default to require.resolve('tslint')
 * @param {String} options.prettierPath - the path to the prettier module.
 *   Will default to require.resovlve('prettier')
 * @param {Object} options.tslintConfig - the config to use for formatting
 *  with TSLint.
 * @param {Object} options.prettierOptions - the options to pass for
 *  formatting with `prettier`. If not provided, prettier-eslint will attempt
 *  to create the options based on the eslintConfig
 * @param {Object} options.fallbackPrettierOptions - the options to pass for
 *  formatting with `prettier` if the given option is not inferrable from the
 *  eslintConfig.
 * @param {Boolean} options.prettierLast - Run Prettier Last
 * @return {String} - the formatted string
 */
function format(options) {
  var filePath = options.filePath;


  var tslintFix = createTSLintFix(options.tslintConfig, options.tslintPath || (0, _utils.getModulePath)(filePath, "tslint"));

  var prettify = createPrettify(options.prettierOptions || options.fallbackPrettierOptions || {}, options.prettierPath || (0, _utils.getModulePath)(filePath, "prettier"));

  var text = options.text || (0, _fs.readFileSync)(filePath, "utf8");
  return options.prettierLast ? prettify(tslintFix(text, filePath), filePath) : tslintFix(prettify(text, filePath), filePath);
}

function createPrettify(formatOptions, prettierPath) {
  var prettier = (0, _utils.requireModule)(prettierPath);
  return function prettify(text, filePath) {
    return prettier.format(text, Object.assign({}, formatOptions, (0, _utils.getPrettierConfig)(filePath), filePath && { filepath: filePath }));
  };
}

function createTSLintFix(defaultLintConfig, tslintPath) {
  var tslint = (0, _utils.requireModule)(tslintPath);
  var findConfiguration = tslint.Configuration.findConfiguration;

  // Adapted from: https://github.com/palantir/tslint/blob/5.12.0/src/linter.ts

  return function tslintFix(text, filePath) {
    // TODO: Use the "fix" option of `new tslint.Linter()` once the following
    // issue is triaged: https://github.com/palantir/tslint/issues/4411
    var linter = new tslint.Linter({
      fix: false, // Disabled to avoid file operations.
      formatter: "json"
    });

    var lintConfig = Object.assign({}, defaultLintConfig, findConfiguration(null, filePath).results);

    linter.lint(filePath, text, lintConfig);

    var _linter$getResult = linter.getResult(),
        failures = _linter$getResult.failures;

    if (!failures.length) {
      return text;
    }

    // This is a private method, but we're using it as a workaround.
    var enabledRules = linter.getEnabledRules(lintConfig, (0, _path.extname)(filePath) === ".js");

    // To keep rules from interfering with one another, we apply their fixes one
    // rule at a time. More info: https://github.com/azz/prettier-tslint/issues/26
    return enabledRules.reduce(function (text, rule) {
      var _rule$getOptions = rule.getOptions(),
          ruleName = _rule$getOptions.ruleName;

      var hasFix = function hasFix(f) {
        return f.hasFix() && f.getRuleName() === ruleName;
      };
      if (failures.some(hasFix)) {
        var sourceFile = tslint.getSourceFile(filePath, text);
        var fixableFailures = tslint.removeDisabledFailures(sourceFile, rule.apply(sourceFile)).filter(function (f) {
          return f.hasFix();
        });

        if (fixableFailures.length) {
          var fixes = fixableFailures.map(function (f) {
            return f.getFix();
          });
          return tslint.Replacement.applyFixes(text, fixes);
        }
      }
      return text;
    }, text);
  };
}