"use strict";
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const parse_util_1 = require("./parse_util");
const html = __importStar(require("./ast"));
const interpolation_config_1 = require("./interpolation_config");
const lex = __importStar(require("./lexer"));
const tags_1 = require("./tags");
const html_tags_1 = require("./html_tags");
class TreeError extends parse_util_1.ParseError {
    constructor(elementName, span, msg) {
        super(span, msg);
        this.elementName = elementName;
    }
    static create(elementName, span, msg) {
        return new TreeError(elementName, span, msg);
    }
}
exports.TreeError = TreeError;
class ParseTreeResult {
    constructor(rootNodes, errors) {
        this.rootNodes = rootNodes;
        this.errors = errors;
    }
}
exports.ParseTreeResult = ParseTreeResult;
class Parser {
    constructor(options = {
        decodeEntities: true,
        ignoreFirstLf: true,
        insertRequiredParents: false,
        selfClosingElements: false,
        selfClosingCustomElements: false
    }, getTagDefinition) {
        this.options = options;
        this.getTagDefinition = getTagDefinition;
    }
    parse(source, url, interpolationConfig = interpolation_config_1.DEFAULT_INTERPOLATION_CONFIG) {
        const tokensAndErrors = lex.tokenize(source, url, this.getTagDefinition, interpolationConfig, this.options);
        const treeAndErrors = new _TreeBuilder(this.options, tokensAndErrors.tokens, this.getTagDefinition).build();
        return new ParseTreeResult(treeAndErrors.rootNodes, tokensAndErrors.errors.concat(treeAndErrors.errors));
    }
}
exports.Parser = Parser;
class _TreeBuilder {
    constructor(options, tokens, getTagDefinition) {
        this.options = options;
        this.tokens = tokens;
        this.getTagDefinition = getTagDefinition;
        this._index = -1;
        this._rootNodes = [];
        this._errors = [];
        this._elementStack = [];
        this._advance();
    }
    build() {
        while (this._peek.type !== lex.TokenType.EOF) {
            if (this._peek.type === lex.TokenType.DOC_TYPE) {
                this._consumeDoctype(this._advance());
            }
            else if (this._peek.type === lex.TokenType.TAG_OPEN_START) {
                this._consumeStartTag(this._advance());
            }
            else if (this._peek.type === lex.TokenType.TAG_CLOSE) {
                this._consumeEndTag(this._advance());
            }
            else if (this._peek.type === lex.TokenType.CDATA_START) {
                this._closeVoidElement();
                this._consumeCdata(this._advance());
            }
            else if (this._peek.type === lex.TokenType.COMMENT_START) {
                this._closeVoidElement();
                this._consumeComment(this._advance());
            }
            else if (this._peek.type === lex.TokenType.TEXT ||
                this._peek.type === lex.TokenType.RAW_TEXT ||
                this._peek.type === lex.TokenType.ESCAPABLE_RAW_TEXT) {
                this._closeVoidElement();
                this._consumeText(this._advance());
            }
            else {
                // Skip all other tokens...
                this._advance();
            }
        }
        return new ParseTreeResult(this._rootNodes, this._errors);
    }
    _advance() {
        const prev = this._peek;
        if (this._index < this.tokens.length - 1) {
            // Note: there is always an EOF token at the end
            this._index++;
        }
        this._peek = this.tokens[this._index];
        return prev;
    }
    _advanceIf(type) {
        if (this._peek.type === type) {
            return this._advance();
        }
        return null;
    }
    _consumeCdata(startToken) {
        this._consumeText(this._advance());
        this._advanceIf(lex.TokenType.CDATA_END);
    }
    _consumeComment(token) {
        const text = this._advanceIf(lex.TokenType.RAW_TEXT);
        this._advanceIf(lex.TokenType.COMMENT_END);
        const value = text != null ? text.parts[0] : null;
        this._addToParent(new html.Comment(value, token.sourceSpan));
    }
    _consumeDoctype(token) {
        const value = token.parts.length ? token.parts[0] : null;
        this._addToParent(new html.Doctype(value, token.sourceSpan));
    }
    _consumeText(token) {
        let text = token.parts[0];
        if (text.length > 0 && text[0] == '\n') {
            const parent = this._getParentElement();
            if (parent != null &&
                parent.children.length == 0 &&
                this.getTagDefinition(parent.name, this.options.ignoreFirstLf, this.options.selfClosingElements)
                    .ignoreFirstLf) {
                text = text.substring(1);
            }
        }
        if (text.length > 0) {
            this._addToParent(new html.Text(text, token.sourceSpan));
        }
    }
    _closeVoidElement() {
        const el = this._getParentElement();
        if (el &&
            this.getTagDefinition(el.name, this.options.ignoreFirstLf, this.options.selfClosingElements).isVoid) {
            this._elementStack.pop();
        }
    }
    _consumeStartTag(startTagToken) {
        const prefix = startTagToken.parts[0];
        const name = startTagToken.parts[1];
        const attrs = [];
        while (this._peek.type === lex.TokenType.ATTR_NAME) {
            attrs.push(this._consumeAttr(this._advance()));
        }
        const nameAndNsInfo = this._getElementNameAndNsInfo(prefix, name, this._getParentElement());
        let selfClosing = false;
        // Note: There could have been a tokenizer error
        // so that we don't get a token for the end tag...
        if (this._peek.type === lex.TokenType.TAG_OPEN_END_VOID) {
            this._advance();
            selfClosing = true;
            const tagDef = this.getTagDefinition(nameAndNsInfo.fullName, this.options.ignoreFirstLf, this.options.selfClosingElements);
            if (!(tagDef.canSelfClose ||
                tags_1.getNsPrefix(nameAndNsInfo.fullName) !== null ||
                tagDef.isVoid ||
                // allow self-closing custom elements
                (this.options.selfClosingCustomElements && html_tags_1.isKnownHTMLTag(nameAndNsInfo.fullName) === false))) {
                this._errors.push(TreeError.create(nameAndNsInfo.fullName, startTagToken.sourceSpan, `Only void, foreign or custom elements can be self closed "${startTagToken.parts[1]}"`));
            }
        }
        else if (this._peek.type === lex.TokenType.TAG_OPEN_END) {
            this._advance();
            selfClosing = false;
        }
        const end = this._peek.sourceSpan.start;
        const span = new parse_util_1.ParseSourceSpan(startTagToken.sourceSpan.start, end);
        const el = new html.Element(nameAndNsInfo.fullName, attrs, [], nameAndNsInfo.implicitNs, span, span, undefined);
        this._pushElement(el);
        if (selfClosing) {
            this._popElement(nameAndNsInfo.fullName);
            el.endSourceSpan = span;
        }
    }
    _pushElement(el) {
        const parentEl = this._getParentElement();
        if (parentEl &&
            this.getTagDefinition(parentEl.name, this.options.ignoreFirstLf, this.options.selfClosingElements).isClosedByChild(el.name)) {
            this._elementStack.pop();
        }
        if (this.options.insertRequiredParents) {
            const tagDef = this.getTagDefinition(el.name, this.options.ignoreFirstLf, this.options.selfClosingElements);
            const { parent, container } = this._getParentElementSkippingContainers();
            if (parent && tagDef.requireExtraParent(parent.name)) {
                const newParent = new html.Element(tagDef.parentToAdd, [], [], el.implicitNs, el.sourceSpan, el.startSourceSpan, el.endSourceSpan);
                this._insertBeforeContainer(parent, container, newParent);
            }
        }
        this._addToParent(el);
        this._elementStack.push(el);
    }
    _consumeEndTag(endTagToken) {
        const nameInfo = this._getElementNameAndNsInfo(endTagToken.parts[0], endTagToken.parts[1], this._getParentElement());
        if (this._getParentElement()) {
            this._getParentElement().endSourceSpan = endTagToken.sourceSpan;
        }
        if (this.getTagDefinition(nameInfo.fullName, this.options.ignoreFirstLf, this.options.selfClosingElements)
            .isVoid) {
            this._errors.push(TreeError.create(nameInfo.fullName, endTagToken.sourceSpan, `Void elements do not have end tags "${endTagToken.parts[1]}"`));
        }
        else if (!this._popElement(nameInfo.fullName)) {
            const errMsg = `Unexpected closing tag "${nameInfo.fullName}". It may happen when the tag has already been closed by another tag. For more info see https://www.w3.org/TR/html5/syntax.html#closing-elements-that-have-implied-end-tags`;
            this._errors.push(TreeError.create(nameInfo.fullName, endTagToken.sourceSpan, errMsg));
        }
    }
    _popElement(fullName) {
        for (let stackIndex = this._elementStack.length - 1; stackIndex >= 0; stackIndex--) {
            const el = this._elementStack[stackIndex];
            if (el.name == fullName) {
                this._elementStack.splice(stackIndex, this._elementStack.length - stackIndex);
                return true;
            }
            if (!this.getTagDefinition(el.name, this.options.ignoreFirstLf, this.options.selfClosingElements)
                .closedByParent) {
                return false;
            }
        }
        return false;
    }
    _consumeAttr(attrName) {
        const fullName = tags_1.mergeNsAndName(attrName.parts[0], attrName.parts[1]);
        let implicitNs = attrName.parts[0] != null;
        let end = attrName.sourceSpan.end;
        let value = '';
        let valueSpan = undefined;
        if (this._peek.type === lex.TokenType.ATTR_VALUE) {
            const valueToken = this._advance();
            value = valueToken.parts[0];
            end = valueToken.sourceSpan.end;
            valueSpan = valueToken.sourceSpan;
        }
        return new html.Attribute(fullName, value, implicitNs, new parse_util_1.ParseSourceSpan(attrName.sourceSpan.start, end), valueSpan);
    }
    _getParentElement() {
        return this._elementStack.length > 0 ? this._elementStack[this._elementStack.length - 1] : null;
    }
    /**
     * Returns the parent in the DOM and the container.
     *
     * `<ng-container>` elements are skipped as they are not rendered as DOM element.
     */
    _getParentElementSkippingContainers() {
        let container = null;
        for (let i = this._elementStack.length - 1; i >= 0; i--) {
            if (!tags_1.isNgContainer(this._elementStack[i].name)) {
                return { parent: this._elementStack[i], container };
            }
            container = this._elementStack[i];
        }
        return { parent: null, container };
    }
    _addToParent(node) {
        const parent = this._getParentElement();
        if (parent != null) {
            parent.children.push(node);
        }
        else {
            this._rootNodes.push(node);
        }
    }
    /**
     * Insert a node between the parent and the container.
     * When no container is given, the node is appended as a child of the parent.
     * Also updates the element stack accordingly.
     *
     * @internal
     */
    _insertBeforeContainer(parent, container, node) {
        if (!container) {
            this._addToParent(node);
            this._elementStack.push(node);
        }
        else {
            if (parent) {
                // replace the container with the new node in the children
                const index = parent.children.indexOf(container);
                parent.children[index] = node;
            }
            else {
                this._rootNodes.push(node);
            }
            node.children.push(container);
            this._elementStack.splice(this._elementStack.indexOf(container), 0, node);
        }
    }
    _getElementNameAndNsInfo(prefix, localName, parentElement) {
        let implicitNs = false;
        if (prefix == null) {
            prefix = this.getTagDefinition(localName, this.options.ignoreFirstLf, this.options.selfClosingElements)
                .implicitNamespacePrefix;
            if (prefix) {
                implicitNs = true;
            }
            if (prefix == null && parentElement != null) {
                prefix = tags_1.getNsPrefix(parentElement.name);
                if (prefix != null) {
                    implicitNs = true;
                }
            }
        }
        return { fullName: tags_1.mergeNsAndName(prefix, localName), implicitNs };
    }
}
function lastOnStack(stack, element) {
    return stack.length > 0 && stack[stack.length - 1] === element;
}
//# sourceMappingURL=parser.js.map