"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.activate = void 0;
const vscode = require("vscode");
const generateGrammarCommand_1 = require("./commands/generateGrammarCommand");
const languages_1 = require("./languages");
const client_1 = require("./client");
const path_1 = require("path");
const virtualFileCommand_1 = require("./commands/virtualFileCommand");
const userSnippetDir_1 = require("./userSnippetDir");
const openUserScaffoldSnippetFolderCommand_1 = require("./commands/openUserScaffoldSnippetFolderCommand");
const doctorCommand_1 = require("./commands/doctorCommand");
const parse_engine_gateway_1 = require("./stml/parse-engine-gateway");
async function activate(context) {
    const isInsiders = vscode.env.appName.includes('Insiders');
    const globalSnippetDir = userSnippetDir_1.getGlobalSnippetDir(isInsiders);
    /**
     * Virtual file display command for debugging template interpolation
     */
    context.subscriptions.push(await virtualFileCommand_1.registerVeturTextDocumentProviders());
    /**
     * Custom Block Grammar generation command
     */
    context.subscriptions.push(vscode.commands.registerCommand('stml.generateGrammar', generateGrammarCommand_1.generateGrammarCommandHandler(context.extensionPath)));
    /**
     * Open custom snippet folder
     */
    context.subscriptions.push(vscode.commands.registerCommand('stml.openUserScaffoldSnippetFolder', openUserScaffoldSnippetFolderCommand_1.generateOpenUserScaffoldSnippetFolderCommand(globalSnippetDir)));
    languages_1.registerLanguageConfigurations();
    // stml 引入外部cssw文件提示
    context.subscriptions.push(registerCompletionProvider('stml', /class=["|']([\w- ]*$)/, "", " "));
    /**
     * Vue Language Server Initialization
     */
    const serverModule = context.asAbsolutePath(path_1.join('server', 'out', 'vueServerMain.js'));
    const client = client_1.initializeLanguageClient(serverModule, globalSnippetDir);
    context.subscriptions.push(client.start());
    const promise = client
        .onReady()
        .then(() => {
        registerCustomClientNotificationHandlers(client);
        registerCustomLSPCommands(context, client);
        registerRestartVLSCommand(context, client);
    })
        .catch(e => {
        console.error(e.stack);
        console.log('Client initialization failed');
    });
    return displayInitProgress(promise);
}
exports.activate = activate;
async function displayInitProgress(promise) {
    return vscode.window.withProgress({
        title: 'Stml initialization',
        location: vscode.ProgressLocation.Window
    }, () => promise);
}
function registerRestartVLSCommand(context, client) {
    context.subscriptions.push(vscode.commands.registerCommand('stml.restartVLS', () => displayInitProgress(client
        .stop()
        .then(() => client.start())
        .then(() => client.onReady()))));
}
function registerCustomClientNotificationHandlers(client) {
    client.onNotification('$/openWebsite', (url) => {
        vscode.env.openExternal(vscode.Uri.parse(url));
    });
    client.onNotification('$/showVirtualFile', (virtualFileSource, prettySourceMap) => {
        virtualFileCommand_1.setVirtualContents(virtualFileSource, prettySourceMap);
    });
}
function registerCustomLSPCommands(context, client) {
    context.subscriptions.push(vscode.commands.registerCommand('stml.showCorrespondingVirtualFile', virtualFileCommand_1.generateShowVirtualFileCommand(client)), vscode.commands.registerCommand('stml.showOutputChannel', () => client.outputChannel.show()), vscode.commands.registerCommand('stml.showDoctorInfo', doctorCommand_1.generateDoctorCommand(client)));
}
const completionTriggerChars = ['"', "'", " ", "."];
const cssPathRegx = /<style\s+src\s*=\s*('|")([^>]*?)('|")[^>]*?/g;
const reg = /^['|"](.*)['|"]$/;
function registerCompletionProvider(languageSelector, classMatchRegex, classPrefix, splitChar) {
    return vscode.languages.registerCompletionItemProvider(languageSelector, {
        provideCompletionItems(document, position) {
            const start = new vscode.Position(position.line, 0);
            const range = new vscode.Range(start, position);
            const text = document.getText(range);
            const allText = document.getText();
            const cssLinks = allText.match(cssPathRegx);
            if (!cssLinks || cssLinks.length === 0) {
                return [];
            }
            const uniqueDefinitions = [];
            cssLinks.forEach(css => {
                const cssUri = css.split('=')[1].replace(reg, "$1");
                const uri = path_1.resolve(path_1.dirname(document.uri.fsPath), cssUri);
                const d = arrDistinctByProp(parse_engine_gateway_1.ParseEngineGateway.callParser(vscode.Uri.file(uri)));
                uniqueDefinitions.push(...d);
            });
            // Check if the cursor is on a class attribute and retrieve all the css rules in this class attribute
            const rawClasses = text.match(classMatchRegex);
            if (!rawClasses || rawClasses.length === 1) {
                return [];
            }
            // Will store the classes found on the class attribute
            const classesOnAttribute = rawClasses[1].split(splitChar);
            // Creates a collection of CompletionItem based on the classes already cached
            const completionItems = uniqueDefinitions.map((definition) => {
                const completionItem = new vscode.CompletionItem(definition.className, vscode.CompletionItemKind.Variable);
                const completionClassName = `${classPrefix}${definition.className}`;
                completionItem.filterText = completionClassName;
                completionItem.insertText = completionClassName;
                return completionItem;
            });
            // Removes from the collection the classes already specified on the class attribute
            for (const classOnAttribute of classesOnAttribute) {
                for (let j = 0; j < completionItems.length; j++) {
                    if (completionItems[j].insertText === classOnAttribute) {
                        completionItems.splice(j, 1);
                    }
                }
            }
            return completionItems;
        }
    }, ...completionTriggerChars);
}
function arrDistinctByProp(arr) {
    let obj = {};
    return arr.reduce((preValue, item) => {
        obj[item['className']] ? '' : obj[item['className']] = true && preValue.push(item);
        return preValue;
    }, []);
}
//# sourceMappingURL=vueMain.js.map